<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CategoryController extends Controller
{
    public function index()
    {
        $categories = Category::withCount('products')
            ->latest()
            ->paginate(10);

        return view('admin.categories.index', compact('categories'));
    }

    public function create()
    {
        return view('admin.categories.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:categories',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:8192',
            // SEO Fields
            'meta_title' => 'nullable|string|max:60',
            'meta_description' => 'nullable|string|max:160',
            'meta_keywords' => 'nullable|string',
            'category_type' => 'nullable|in:product,service,blog',
            'sort_order' => 'nullable|integer|min:0',
            'seo_url' => 'nullable|string|max:255',
            'seo_priority' => 'nullable|in:low,normal,high',
            'seo_indexable' => 'boolean',
            'show_in_menu' => 'boolean',
            'show_in_sitemap' => 'boolean',
        ]);

        $data = $request->all();

        // Handle image upload
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_' . $image->getClientOriginalName();

            // Create directory if it doesn't exist
            $uploadPath = public_path('uploads/categories');
            if (!file_exists($uploadPath)) {
                mkdir($uploadPath, 0755, true);
            }

            // Move uploaded file
            $image->move($uploadPath, $imageName);
            $data['image'] = 'uploads/categories/' . $imageName;
        }

        // Generate slug from name
        $data['slug'] = Str::slug($request->name);

        // Set default values
        $data['seo_indexable'] = $request->boolean('seo_indexable', true);
        $data['show_in_menu'] = $request->boolean('show_in_menu', true);
        $data['show_in_sitemap'] = $request->boolean('show_in_sitemap', true);
        $data['category_type'] = $request->category_type ?: 'product';
        $data['seo_priority'] = $request->seo_priority ?: 'normal';
        $data['sort_order'] = $request->sort_order ?: 0;

        $category = Category::create($data);

        // Auto-generate SEO fields if needed
        $category->autoGenerateSeoFields();

        return redirect()
            ->route('admin.categories.index')
            ->with('success', 'Category created successfully.');
    }

    public function edit(Category $category)
    {
        return view('admin.categories.edit', compact('category'));
    }

    public function update(Request $request, Category $category)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:categories,name,' . $category->id,
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:8192',
            // SEO Fields
            'meta_title' => 'nullable|string|max:60',
            'meta_description' => 'nullable|string|max:160',
            'meta_keywords' => 'nullable|string',
            'category_type' => 'nullable|in:product,service,blog',
            'sort_order' => 'nullable|integer|min:0',
            'seo_url' => 'nullable|string|max:255',
            'seo_priority' => 'nullable|in:low,normal,high',
            'seo_indexable' => 'boolean',
            'show_in_menu' => 'boolean',
            'show_in_sitemap' => 'boolean',
        ]);

        $data = $request->all();

        // Handle remove image
        if ($request->has('remove_image') && $category->image && file_exists(public_path($category->image))) {
            unlink(public_path($category->image));
            $data['image'] = null;
        }

        // Handle image upload
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_' . $image->getClientOriginalName();

            // Create directory if it doesn't exist
            $uploadPath = public_path('uploads/categories');
            if (!file_exists($uploadPath)) {
                mkdir($uploadPath, 0755, true);
            }

            // Move uploaded file
            $image->move($uploadPath, $imageName);
            $data['image'] = 'uploads/categories/' . $imageName;
        }

        // Generate slug from name
        $data['slug'] = Str::slug($request->name);

        // Set default values
        $data['seo_indexable'] = $request->boolean('seo_indexable', true);
        $data['show_in_menu'] = $request->boolean('show_in_menu', true);
        $data['show_in_sitemap'] = $request->boolean('show_in_sitemap', true);
        $data['category_type'] = $request->category_type ?: 'product';
        $data['seo_priority'] = $request->seo_priority ?: 'normal';
        $data['sort_order'] = $request->sort_order ?: 0;

        $category->update($data);

        // Auto-generate SEO fields if needed
        $category->autoGenerateSeoFields();

        return redirect()
            ->route('admin.categories.index')
            ->with('success', 'Category updated successfully.');
    }

    public function destroy(Category $category)
    {
        if ($category->products()->count() > 0) {
            return redirect()
                ->route('admin.categories.index')
                ->with('error', 'Cannot delete category with associated products.');
        }

        // Delete image file if exists
        if ($category->image && file_exists(public_path($category->image))) {
            unlink(public_path($category->image));
        }

        $category->delete();

        return redirect()
            ->route('admin.categories.index')
            ->with('success', 'Category deleted successfully.');
    }
}
