<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class OrderController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $orders = Order::with(['user', 'items.product'])
            ->latest()
            ->paginate(10);

        return view('admin.orders.index', compact('orders'));
    }

    /**
     * Display approved orders ready for shipping
     */
    public function approved()
    {
        $orders = Order::with(['user', 'items.product'])
            ->where('status', Order::STATUS_APPROVED)
            ->latest()
            ->paginate(10);

        return view('admin.orders.approved', compact('orders'));
    }

    /**
     * Display shipped orders
     */
    public function shipped()
    {
        $orders = Order::with(['user', 'items.product'])
            ->whereIn('status', [Order::STATUS_SHIPPED, Order::STATUS_DELIVERED])
            ->latest()
            ->paginate(10);

        return view('admin.orders.shipped', compact('orders'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(Order $order)
    {
        $order->load(['user', 'items.product']);

        // Calculate totals including additional prices, handling deleted products
        $subtotal = $order->items->sum(function ($item) {
            return $item->price * $item->quantity;
        });

        $tax = $subtotal * Setting::getTaxRate();
        $total = $subtotal + $tax;

        return view('admin.orders.show', compact('order', 'subtotal', 'tax', 'total'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }

    public function updateStatus(Request $request, Order $order)
    {
        $request->validate([
            'status' => 'required|in:pending,approved,shipped,delivered,cancelled'
        ]);

        $order->update([
            'status' => $request->status
        ]);

        return redirect()->back()->with('success', 'Order status updated successfully.');
    }

    /**
     * Mark order as shipped with tracking number
     */
    public function markAsShipped(Request $request, Order $order)
    {
        $request->validate([
            'tracking_number' => 'required|string|max:255'
        ]);

        $order->update([
            'status' => Order::STATUS_SHIPPED,
            'tracking_number' => $request->tracking_number,
            'shipping_date' => now(),
        ]);

        return redirect()->back()->with('success', 'Order marked as shipped successfully.');
    }

    /**
     * Mark order as delivered
     */
    public function markAsDelivered(Order $order)
    {
        $order->update([
            'status' => Order::STATUS_DELIVERED,
            'delivery_date' => now(),
        ]);

        return redirect()->back()->with('success', 'Order marked as delivered successfully.');
    }

    /**
     * Approve order for shipping
     */
    public function approveOrder(Order $order)
    {
        $order->update([
            'status' => Order::STATUS_APPROVED,
        ]);

        return redirect()->back()->with('success', 'Order approved for shipping successfully.');
    }
}
