<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Page;
use Illuminate\Http\Request;

class PageController extends Controller
{
    public function index()
    {
        $pages = Page::latest()->paginate(10);
        return view('admin.pages.index', compact('pages'));
    }

    public function create()
    {
        return view('admin.pages.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:pages',
            'content' => 'nullable|string',
            'meta_title' => 'nullable|string|max:60',
            'meta_description' => 'nullable|string|max:160',
            'meta_keywords' => 'nullable|string|max:255',
            'og_title' => 'nullable|string|max:60',
            'og_description' => 'nullable|string|max:160',
            'og_image' => 'nullable|string|max:255',
            'twitter_title' => 'nullable|string|max:60',
            'twitter_description' => 'nullable|string|max:160',
            'twitter_image' => 'nullable|string|max:255',
            'canonical_url' => 'nullable|string|max:255',
            'robots_directive' => 'nullable|string|max:50',
            'seo_priority' => 'nullable|in:low,normal,high',
            'seo_indexable' => 'boolean',
            'show_in_sitemap' => 'boolean',
            'show_in_menu' => 'boolean',
        ]);

        $data = $request->all();
        $data['is_active'] = $request->boolean('is_active', false);
        $data['seo_indexable'] = $request->boolean('seo_indexable', true);
        $data['show_in_sitemap'] = $request->boolean('show_in_sitemap', true);
        $data['show_in_menu'] = $request->boolean('show_in_menu', false);
        $data['seo_priority'] = $request->seo_priority ?: 'normal';
        $data['robots_directive'] = $request->robots_directive ?: 'index,follow';

        Page::create($data);
        return redirect()->route('admin.pages.index')->with('success', 'Page created successfully.');
    }

    public function edit(Page $page)
    {
        return view('admin.pages.edit', compact('page'));
    }

    public function update(Request $request, Page $page)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:pages,slug,' . $page->id,
            'content' => 'nullable|string',
            'meta_title' => 'nullable|string|max:60',
            'meta_description' => 'nullable|string|max:160',
            'meta_keywords' => 'nullable|string|max:255',
            'og_title' => 'nullable|string|max:60',
            'og_description' => 'nullable|string|max:160',
            'og_image' => 'nullable|string|max:255',
            'twitter_title' => 'nullable|string|max:60',
            'twitter_description' => 'nullable|string|max:160',
            'twitter_image' => 'nullable|string|max:255',
            'canonical_url' => 'nullable|string|max:255',
            'robots_directive' => 'nullable|string|max:50',
            'seo_priority' => 'nullable|in:low,normal,high',
            'seo_indexable' => 'boolean',
            'show_in_sitemap' => 'boolean',
            'show_in_menu' => 'boolean',
        ]);

        $data = $request->all();
        $data['is_active'] = $request->boolean('is_active', false);
        $data['seo_indexable'] = $request->boolean('seo_indexable', true);
        $data['show_in_sitemap'] = $request->boolean('show_in_sitemap', true);
        $data['show_in_menu'] = $request->boolean('show_in_menu', false);
        $data['seo_priority'] = $request->seo_priority ?: 'normal';
        $data['robots_directive'] = $request->robots_directive ?: 'index,follow';

        $page->update($data);
        return redirect()->route('admin.pages.index')->with('success', 'Page updated successfully.');
    }

    public function destroy(Page $page)
    {
        $page->delete();
        return redirect()->route('admin.pages.index')->with('success', 'Page deleted successfully.');
    }

    public function toggle(Page $page)
    {
        $page->is_active = !$page->is_active;
        $page->save();
        return redirect()->route('admin.pages.index')->with('success', 'Page status updated.');
    }
}
