<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\NewsletterController;
use App\Mail\ProductNewsletter;
use Illuminate\Support\Facades\Mail;
use App\Models\NewsletterSubscriber;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $products = Product::with(['category', 'admin'])
            ->latest()
            ->paginate(10);

        return view('admin.products.index', compact('products'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = Category::all();
        return view('admin.products.create', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Debug: Log the entire request
        Log::info('Request Data:', [
            'all' => $request->all(),
            'has_dynamic_fields' => $request->has('dynamic_fields'),
            'dynamic_fields' => $request->dynamic_fields
        ]);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'price' => 'required|numeric|min:0',
            'shipping_cost' => 'nullable|numeric|min:0',
            'stock' => 'required|integer|min:0',
            'category_id' => 'required|exists:categories,id',
            'images.*' => 'image|mimes:jpeg,png,jpg,gif|max:2048',
            'dynamic_fields' => 'nullable|string',
            'is_featured' => 'boolean',
            'discount_percentage' => 'nullable|integer|min:0|max:100',
            // SEO Fields
            'meta_title' => 'nullable|string|max:60',
            'meta_description' => 'nullable|string|max:160',
            'meta_keywords' => 'nullable|string',
            'brand' => 'nullable|string|max:255',
            'model' => 'nullable|string|max:255',
            'sku' => 'nullable|string|max:255',
            'gtin' => 'nullable|string|max:255',
            'mpn' => 'nullable|string|max:255',
            'availability' => 'nullable|in:in_stock,out_of_stock,preorder',
            'condition' => 'nullable|in:new,used,refurbished',
            'seo_url' => 'nullable|string|max:255',
            'seo_priority' => 'nullable|in:low,normal,high',
            'seo_indexable' => 'boolean',
        ]);

        try {
            DB::beginTransaction();

            // Create unique slug from name
            $baseSlug = Str::slug($validated['name']);
            $slug = $baseSlug;
            $counter = 1;

            // Check if slug exists and append number if it does
            while (Product::where('slug', $slug)->exists()) {
                $slug = $baseSlug . '-' . $counter;
                $counter++;
            }

            // Handle image uploads
            $images = [];
            if ($request->hasFile('images')) {
                // Create directory if it doesn't exist
                $uploadPath = public_path('uploads/products');
                if (!file_exists($uploadPath)) {
                    mkdir($uploadPath, 0755, true);
                }

                foreach ($request->file('images') as $image) {
                    $fileName = time() . '_' . $image->getClientOriginalName();
                    $image->move($uploadPath, $fileName);
                    $images[] = 'uploads/products/' . $fileName;
                }
            }

            // Process dynamic fields
            $dynamicFields = null;
            if ($request->has('dynamic_fields') && !empty($request->dynamic_fields)) {
                $dynamicFields = $request->dynamic_fields;
                // Debug: Log the dynamic fields value
                Log::info('Dynamic Fields Value:', [
                    'raw' => $dynamicFields,
                    'decoded' => json_decode($dynamicFields, true)
                ]);
            }

            // Create the product
            $productData = [
                'name' => $validated['name'],
                'slug' => $slug,
                'description' => $validated['description'],
                'price' => $validated['price'],
                'shipping_cost' => $request->filled('shipping_cost') ? $request->shipping_cost : null,
                'stock' => $validated['stock'],
                'category_id' => $validated['category_id'],
                'admin_id' => auth('admin')->id(),
                'is_featured' => $request->boolean('is_featured', false),
                'discount_percentage' => $request->discount_percentage,
                'is_approved' => true,
                'image' => !empty($images) ? json_encode($images) : null,
                // SEO Fields
                'meta_title' => $request->meta_title,
                'meta_description' => $request->meta_description,
                'meta_keywords' => $request->meta_keywords,
                'brand' => $request->brand,
                'model' => $request->model,
                'sku' => $request->sku,
                'gtin' => $request->gtin,
                'mpn' => $request->mpn,
                'availability' => $request->availability ?: 'in_stock',
                'condition' => $request->condition ?: 'new',
                'seo_url' => $request->seo_url ?: $slug,
                'seo_priority' => $request->seo_priority ?: 'normal',
                'seo_indexable' => $request->boolean('seo_indexable', true),
            ];

            // Add dynamic fields if they exist
            if ($dynamicFields !== null) {
                $productData['dynamic_fields'] = $dynamicFields;
            }

            // Debug: Log the final product data
            Log::info('Final Product Data:', $productData);

            $product = Product::create($productData);

            // Auto-generate SEO fields if requested
            if ($request->boolean('auto_generate_seo', true)) {
                $product->autoGenerateSeoFields();
            }

            // Send newsletter notification
            try {
                $subscribers = NewsletterSubscriber::pluck('email')->toArray();
                if (!empty($subscribers)) {
                    Mail::to($subscribers)->send(new ProductNewsletter($product, 'new'));
                }
            } catch (\Exception $e) {
                Log::error('Newsletter send error: ' . $e->getMessage());
            }

            // Debug: Log the created product
            Log::info('Created Product:', [
                'id' => $product->id,
                'dynamic_fields' => $product->dynamic_fields,
                'raw_dynamic_fields' => $product->getRawOriginal('dynamic_fields')
            ]);

            DB::commit();

            return redirect()
                ->route('admin.products.index')
                ->with('success', 'Product created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Product Creation Error:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return back()
                ->withInput()
                ->with('error', 'Error creating product: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Product $product)
    {
        $categories = Category::all();
        return view('admin.products.edit', compact('product', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Product $product)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'price' => 'required|numeric|min:0',
            'shipping_cost' => 'nullable|numeric|min:0',
            'stock' => 'required|integer|min:0',
            'category_id' => 'required|exists:categories,id',
            'images.*' => 'image|mimes:jpeg,png,jpg,gif|max:2048',
            'dynamic_fields' => 'nullable|string',
            'is_featured' => 'boolean',
            'discount_percentage' => 'nullable|integer|min:0|max:100',
            // SEO Fields
            'meta_title' => 'nullable|string|max:60',
            'meta_description' => 'nullable|string|max:160',
            'meta_keywords' => 'nullable|string',
            'brand' => 'nullable|string|max:255',
            'model' => 'nullable|string|max:255',
            'sku' => 'nullable|string|max:255',
            'gtin' => 'nullable|string|max:255',
            'mpn' => 'nullable|string|max:255',
            'availability' => 'nullable|in:in_stock,out_of_stock,preorder',
            'condition' => 'nullable|in:new,used,refurbished',
            'seo_url' => 'nullable|string|max:255',
            'seo_priority' => 'nullable|in:low,normal,high',
            'seo_indexable' => 'boolean',
        ]);

        try {
            DB::beginTransaction();

            // Update slug if name changed
            if ($product->name !== $validated['name']) {
                $baseSlug = Str::slug($validated['name']);
                $slug = $baseSlug;
                $counter = 1;

                // Check if slug exists (excluding current product) and append number if it does
                while (Product::where('slug', $slug)->where('id', '!=', $product->id)->exists()) {
                    $slug = $baseSlug . '-' . $counter;
                    $counter++;
                }
                $validated['slug'] = $slug;
            }

            // Handle image uploads
            $images = json_decode($product->image ?? '[]', true);
            // Handle remove_images (delete selected images)
            if ($request->has('remove_images')) {
                $removeImages = $request->input('remove_images', []);
                foreach ($removeImages as $removeImage) {
                    $removePath = public_path($removeImage);
                    if (file_exists($removePath)) {
                        unlink($removePath);
                    }
                    // Remove from images array
                    if (($key = array_search($removeImage, $images)) !== false) {
                        unset($images[$key]);
                    }
                }
                $images = array_values($images); // reindex
            }
            if ($request->hasFile('images')) {
                // Create directory if it doesn't exist
                $uploadPath = public_path('uploads/products');
                if (!file_exists($uploadPath)) {
                    mkdir($uploadPath, 0755, true);
                }
                // Upload new images and append to existing
                foreach ($request->file('images') as $image) {
                    $fileName = time() . '_' . $image->getClientOriginalName();
                    $image->move($uploadPath, $fileName);
                    $images[] = 'uploads/products/' . $fileName;
                }
            }

            // Process dynamic fields
            $dynamicFields = null;
            if ($request->has('dynamic_fields') && !empty($request->dynamic_fields)) {
                $dynamicFields = $request->dynamic_fields;
                // Validate JSON format
                if (json_decode($dynamicFields) === null) {
                    throw new \Exception('Invalid dynamic fields format');
                }
            }

            // Update the product
            $updateData = [
                'name' => $validated['name'],
                'description' => $validated['description'],
                'price' => $validated['price'],
                'shipping_cost' => $request->filled('shipping_cost') ? $request->shipping_cost : null,
                'stock' => $validated['stock'],
                'category_id' => $validated['category_id'],
                'dynamic_fields' => $dynamicFields,
                'is_featured' => $request->boolean('is_featured', false),
                'discount_percentage' => $request->discount_percentage,
                'image' => !empty($images) ? json_encode($images) : null,
                // SEO Fields
                'meta_title' => $request->meta_title,
                'meta_description' => $request->meta_description,
                'meta_keywords' => $request->meta_keywords,
                'brand' => $request->brand,
                'model' => $request->model,
                'sku' => $request->sku,
                'gtin' => $request->gtin,
                'mpn' => $request->mpn,
                'availability' => $request->availability ?: 'in_stock',
                'condition' => $request->condition ?: 'new',
                'seo_url' => $request->seo_url ?: $validated['slug'] ?? $product->slug,
                'seo_priority' => $request->seo_priority ?: 'normal',
                'seo_indexable' => $request->boolean('seo_indexable', true),
            ];

            // Add slug if it was updated
            if (isset($validated['slug'])) {
                $updateData['slug'] = $validated['slug'];
            }

            $product->update($updateData);

            // Auto-generate SEO fields if requested
            if ($request->boolean('auto_generate_seo', true)) {
                $product->autoGenerateSeoFields();
            }

            // Send newsletter notification for update
            try {
                $subscribers = NewsletterSubscriber::pluck('email')->toArray();
                if (!empty($subscribers)) {
                    Mail::to($subscribers)->send(new ProductNewsletter($product, 'update'));
                }
            } catch (\Exception $e) {
                Log::error('Newsletter send error: ' . $e->getMessage());
            }

            DB::commit();

            return redirect()
                ->route('admin.products.index')
                ->with('success', 'Product updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()
                ->withInput()
                ->with('error', 'Error updating product: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        // Delete product images
        $images = json_decode($product->image, true) ?? [];
        foreach ($images as $image) {
            if (file_exists(public_path($image))) {
                unlink(public_path($image));
            }
        }

        $product->delete();

        return redirect()
            ->route('admin.products.index')
            ->with('success', 'Product deleted successfully.');
    }

    public function deleteImage(Request $request, Product $product)
    {
        $request->validate([
            'image' => 'required|string'
        ]);

        $imagePath = $request->image;

        // Get current images from the product
        $images = json_decode($product->image ?? '[]', true);

        // Find and remove the image from the array
        $imageIndex = array_search($imagePath, $images);

        if ($imageIndex !== false) {
            // Delete the physical file
            if (file_exists(public_path($imagePath))) {
                unlink(public_path($imagePath));
            }

            // Remove the image from the array
            unset($images[$imageIndex]);
            $images = array_values($images); // Re-index array

            // Update the product with the new images array
            $product->update(['image' => !empty($images) ? json_encode($images) : null]);

            return response()->json(['success' => true]);
        }

        return response()->json(['success' => false, 'message' => 'Image not found'], 404);
    }
}
