<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Cart;
use App\Models\Setting;
use Illuminate\Support\Facades\Auth;

class CartController extends Controller
{
    public function add(Request $request, $productId)
    {
        $product = Product::findOrFail($productId);
        $quantity = $request->input('quantity', 1);
        $totalPrice = $request->input('total_price', $product->price);
        $dynamicSelections = $request->input('dynamic_selections', []);

        // Convert dynamic_selections to JSON if it's a string
        if (is_string($dynamicSelections)) {
            $decoded = json_decode($dynamicSelections, true);
            $dynamicSelections = is_array($decoded) ? $decoded : [];
        }

        // Calculate additional price
        $additionalPrice = 0;
        if (!empty($dynamicSelections)) {
            $additionalPrice = $totalPrice - $product->price;
        }

        // Check if product is already in cart
        $cartItem = Cart::where('user_id', Auth::id())
            ->where('product_id', $productId)
            ->first();

        if ($cartItem) {
            // Update quantity if product exists in cart
            $cartItem->quantity += $quantity;
            $cartItem->dynamic_selections = $dynamicSelections;
            $cartItem->additional_price = $additionalPrice;
            $cartItem->total_price = $totalPrice;
            $cartItem->save();
        } else {
            // Add new item to cart
            Cart::create([
                'user_id' => Auth::id(),
                'product_id' => $productId,
                'quantity' => $quantity,
                'dynamic_selections' => $dynamicSelections,
                'additional_price' => $additionalPrice,
                'total_price' => $totalPrice
            ]);
        }

        return redirect()->back()->with('success', 'Product added to cart successfully.');
    }

    public function index()
    {
        $cartItems = Cart::with('product')
            ->where('user_id', Auth::id())
            ->get();

        // Calculate subtotal (without tax)
        $subtotal = $cartItems->sum(function ($item) {
            $itemPrice = $item->total_price ?? $item->product->price;
            return $item->quantity * $itemPrice;
        });

        // Calculate total shipping cost
        $shippingTotal = $cartItems->sum(function ($item) {
            return $item->product->shipping_cost ? ($item->product->shipping_cost * $item->quantity) : 0;
        });

        // Calculate tax (8%)
        $tax = $subtotal * Setting::getTaxRate();

        // Calculate total with tax and shipping
        $total = $subtotal + $shippingTotal + $tax;

        return view('cart.index', compact('cartItems', 'subtotal', 'tax', 'total', 'shippingTotal'));
    }

    public function update(Request $request, $id)
    {
        $cartItem = Cart::where('user_id', Auth::id())
            ->where('id', $id)
            ->firstOrFail();

        $quantity = $request->input('quantity');
        if ($quantity > 0) {
            $cartItem->quantity = $quantity;

            // Recalculate total_price based on quantity and additional_price
            $basePrice = $cartItem->product->price;
            $additionalPrice = $cartItem->additional_price ?? 0;
            $cartItem->total_price = $basePrice + $additionalPrice;

            $cartItem->save();
        }

        return redirect()->back()->with('success', 'Cart updated successfully.');
    }

    public function remove($id)
    {
        $cartItem = Cart::where('user_id', Auth::id())
            ->where('id', $id)
            ->firstOrFail();

        $cartItem->delete();

        return redirect()->back()->with('success', 'Item removed from cart successfully.');
    }
}
