<?php

namespace App\Http\Controllers;

use App\Models\ChatRoom;
use App\Services\ChatFileService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Events\ChatMessageSent;
use Illuminate\Support\Facades\Log;

class ChatController extends Controller
{
    // User sees or creates their chat
    public function userChat()
    {
        // First, try to find any existing chat for this user (active or closed)
        $existingChat = ChatRoom::where('user_id', Auth::id())->first();

        if ($existingChat) {
            // If chat exists but is closed, reactivate it
            if (!$existingChat->is_active) {
                $existingChat->is_active = true;
                $existingChat->admin_id = null; // Remove previous admin assignment
                $existingChat->save();
            }
            $chat = $existingChat;
        } else {
            // Create new chat only if user has never chatted before
            $chat = ChatRoom::create([
                'user_id' => Auth::id(),
                'admin_id' => null,
                'is_active' => true
            ]);
        }

        return redirect()->route('chat.show', $chat);
    }

    // Admin sees list of available chats and their assigned chats
    public function adminChats()
    {
        // دیگر پیام‌های خوانده‌نشده را به محض ورود ادمین به صفحه، خوانده نکن!
        // فقط لیست چت‌ها را نمایش بده و unread badge در view نمایش داده شود
        $chats = ChatRoom::with(['user', 'admin'])
            ->select('*')
            ->whereIn('id', function($query) {
                $query->selectRaw('MAX(id)')
                      ->from('chat_rooms')
                      ->groupBy('user_id');
            })
            ->orderBy('updated_at', 'desc')
            ->get();

        $fileService = app(\App\Services\ChatFileService::class);
        $total_unread_count = 0;
        foreach ($chats as $chat) {
            $messages = $fileService->getMessages($chat->id);
            $chat->message_count = count($messages);
            $chat->user_message_count = collect($messages)->where('sender_type', 'user')->count();
            $chat->unread_count = collect($messages)->where('sender_type', 'user')->whereIn('is_read', [false, 0, '0', 'false'])->count();
            $chat->unread_count_file = $chat->unread_count;
            $chat->last_message = collect($messages)->sortByDesc('id')->first();
            $total_unread_count += $chat->unread_count;
        }

        return view('chat.admin_chats', compact('chats', 'total_unread_count'));
    }

    // Refresh admin chats (AJAX)
    public function refreshChats()
    {
        $chats = ChatRoom::with(['user', 'admin'])
            ->select('*')
            ->whereIn('id', function($query) {
                $query->selectRaw('MAX(id)')
                      ->from('chat_rooms')
                      ->groupBy('user_id');
            })
            ->orderBy('updated_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'chats' => $chats,
            'available_count' => $chats->where('admin_id', null)->where('is_active', true)->count(),
            'my_active_count' => $chats->where('admin_id', auth('admin')->id())->where('is_active', true)->count(),
            'other_admin_count' => $chats->where('is_active', true)->where('admin_id', '!=', null)->where('admin_id', '!=', auth('admin')->id())->count(),
            'closed_count' => $chats->where('is_active', false)->count()
        ]);
    }

    // Admin assigns a chat to themselves
    public function assignAdmin(ChatRoom $chat)
    {
        if ($chat->admin_id === null) {
            $chat->admin_id = Auth::guard('admin')->id();
            $chat->save();
        }
        return redirect()->route('admin.chat.view', $chat);
    }

    // Show chat page
    public function show(ChatRoom $chat)
    {
        // Chat lock logic (only for active chats, with real-time presence)
        if ($chat->is_active) {
            $lockKey = "chat_{$chat->id}_lock";
            $currentAdminId = Auth::guard('admin')->id();
            $lockedBy = cache()->get($lockKey);
            if ($lockedBy && $lockedBy != $currentAdminId) {
                abort(403, 'This chat is currently being handled by another admin.');
            }
            // Set or refresh lock for 20 seconds (will be kept alive by JS ping)
            cache()->put($lockKey, $currentAdminId, 20);
        }
        $messages = app(ChatFileService::class)->getMessages($chat->id);
        $chat->load(['user', 'admin']);
        $adminTyping = false;
        $userTyping = false;
        if ($chat->admin_id) {
            $typingKey = "chat_{$chat->id}_typing_admin_{$chat->admin_id}";
            $adminTyping = cache()->has($typingKey);
        }
        $typingKey = "chat_{$chat->id}_typing_user_{$chat->user_id}";
        $userTyping = cache()->has($typingKey);
        return view('chat.show', compact('chat', 'messages', 'adminTyping', 'userTyping'));
    }

    // Handle typing events
    public function typing(Request $request, ChatRoom $chat)
    {
        // Remove strict validation to prevent 422 errors
        $isTyping = $request->has('typing') ? (bool) $request->typing : false;

        // Users can only send typing events as users
        if (Auth::check() && $chat->user_id == Auth::id()) {
            $senderType = 'user';
            $senderId = Auth::id();
        }
        // Admins can only send typing events as admins
        elseif (Auth::guard('admin')->check()) {
            // Only assigned admin can send typing events
            if ($chat->admin_id == Auth::guard('admin')->id()) {
                $senderType = 'admin';
                $senderId = Auth::guard('admin')->id();
            } else {
                abort(403, 'You are not assigned to this chat');
            }
        } else {
            abort(403, 'Unauthorized');
        }

        // Store typing status in cache for real-time updates
        $typingKey = "chat_{$chat->id}_typing_{$senderType}_{$senderId}";

        if ($isTyping) {
            cache()->put($typingKey, true, 10); // Expire after 10 seconds
        } else {
            cache()->forget($typingKey);
        }

        // Always return JSON for AJAX requests
        return response()->json([
            'success' => true,
            'typing' => $isTyping,
            'sender_type' => $senderType
        ]);
    }

    // Send message
    public function sendMessage(Request $request, ChatRoom $chat)
    {
        $request->validate(['message' => 'required|string']);
        if (Auth::check() && $chat->user_id == Auth::id()) {
            $senderType = 'user';
            $senderId = Auth::id();
            if (!$chat->is_active) {
                $chat->is_active = true;
                $chat->admin_id = null;
                $chat->save();
            }
        } elseif (Auth::guard('admin')->check()) {
            // Remove assignment restriction: allow any admin
            $senderType = 'admin';
            $senderId = Auth::guard('admin')->id();
            if ($chat->admin_id === null) {
                $chat->admin_id = Auth::guard('admin')->id();
                $chat->save();
            }
            if (!$chat->is_active) {
                $chat->is_active = true;
                $chat->save();
            }
        } else {
            abort(403, 'Unauthorized');
        }
        $typingKey = "chat_{$chat->id}_typing_{$senderType}_{$senderId}";
        cache()->forget($typingKey);
        $messages = app(ChatFileService::class)->getMessages($chat->id);
        $newId = empty($messages) ? 1 : (max(array_column($messages, 'id')) + 1);
        $now = now()->toDateTimeString();
        $message = [
            'id' => $newId,
            'chat_room_id' => $chat->id,
            'sender_type' => $senderType,
            'sender_id' => $senderId,
            'message' => $request->message,
            'is_read' => false,
            'delivery_status' => 'sent',
            'created_at' => $now,
            'updated_at' => $now,
        ];
        if ($senderType === 'admin') {
            $admin = \App\Models\Admin::find($senderId);
            if ($admin) {
                $message['admin_id'] = $senderId;
                $message['admin_name'] = $admin->name;
                $message['sender_name'] = $admin->name;
            } else {
                // اگر ادمین پیدا نشد، شاید این ایدی مربوط به یوزر باشد
                $user = \App\Models\User::find($senderId);
                if ($user) {
                    $message['admin_id'] = $senderId;
                    $message['admin_name'] = $user->name;
                    $message['sender_name'] = $user->name;
                } else {
                    $message['admin_id'] = $senderId;
                    $message['admin_name'] = 'Admin';
                    $message['sender_name'] = 'Admin';
                }
            }
        } else {
            $message['sender_name'] = $chat->user->name;
        }
        $messages[] = $message;
        app(ChatFileService::class)->updateMessages($chat->id, $messages);
        event(new ChatMessageSent($chat->id, $message));
        return response()->json([
            'success' => true,
            'message' => $message,
            'sender_name' => $message['sender_name'],
            'status' => 'Message sent successfully',
            'chat_reactivated' => $chat->is_active
        ], 200, [], JSON_UNESCAPED_UNICODE);
    }

    // Mark messages as read
    public function markAsRead(ChatRoom $chat)
    {
        $isAdmin = Auth::guard('admin')->check(); // Allow any admin
        $isUser = Auth::check() && $chat->user_id == Auth::id();

        if (!$isAdmin && !$isUser) {
            Log::warning('MarkAsRead unauthorized', [
                'chat_id' => $chat->id,
                'auth_user_id' => Auth::id(),
                'auth_admin_id' => Auth::guard('admin')->id(),
            ]);
            return response()->json(['success' => false, 'error' => 'Unauthorized'], 403);
        }

        $referer = request()->headers->get('referer');
        $expectedUserUrl = url("/chat/{$chat->id}");
        $expectedAdminUrl = url("/admin/chat/{$chat->id}/view");
        $isFromChatPage = false;
        if ($isUser && $referer && strpos($referer, $expectedUserUrl) === 0) {
            $isFromChatPage = true;
        } elseif ($isAdmin && $referer && strpos($referer, $expectedAdminUrl) === 0) {
            $isFromChatPage = true;
        }
        if (!$isFromChatPage) {
            Log::info('MarkAsRead blocked: not from chat page', [
                'chat_id' => $chat->id,
                'referer' => $referer,
                'expectedUserUrl' => $expectedUserUrl,
                'expectedAdminUrl' => $expectedAdminUrl,
                'isUser' => $isUser,
                'isAdmin' => $isAdmin,
            ]);
            return response()->json(['success' => false, 'error' => 'Not from chat page'], 403);
        }

        $messages = app(ChatFileService::class)->getMessages($chat->id);
        $updatedMessageIds = [];
        $now = now()->toDateTimeString();
        if ($isUser) {
            foreach ($messages as &$msg) {
                if ($msg['sender_type'] === 'admin' && $msg['delivery_status'] !== 'read') {
                    $msg['is_read'] = true;
                    $msg['delivery_status'] = 'read';
                    $msg['updated_at'] = $now;
                    $updatedMessageIds[] = $msg['id'];
                }
            }
            unset($msg);
        }
        if ($isAdmin) {
            foreach ($messages as &$msg) {
                if ($msg['sender_type'] === 'user' && $msg['delivery_status'] !== 'read') {
                    $msg['is_read'] = true;
                    $msg['delivery_status'] = 'read';
                    $msg['updated_at'] = $now;
                    $updatedMessageIds[] = $msg['id'];
                }
            }
            unset($msg);
        }
        app(ChatFileService::class)->updateMessages($chat->id, $messages);

        Log::info('MarkAsRead called', [
            'chat_id' => $chat->id,
            'isAdmin' => $isAdmin,
            'isUser' => $isUser,
            'auth_user_id' => Auth::id(),
            'auth_admin_id' => Auth::guard('admin')->id(),
            'updated_ids' => $updatedMessageIds,
        ]);

        return response()->json([
            'success' => true,
            'updated_message_ids' => $updatedMessageIds
        ], 200, [], JSON_UNESCAPED_UNICODE);
    }

    // Close chat by admin
    public function close(ChatRoom $chat)
    {
        if (Auth::guard('admin')->check()) {
            $chat->is_active = false;
            $chat->admin_id = Auth::guard('admin')->id(); // Always set to closing admin
            $chat->save();

            // If AJAX request, return JSON
            if (request()->ajax()) {
                return response()->json([
                    'success' => true,
                    'status' => 'Chat closed successfully'
                ], 200, [], JSON_UNESCAPED_UNICODE);
            }
        }
        return redirect()->route('admin.chats');
    }

    // Reopen chat by admin
    public function reopen(ChatRoom $chat)
    {
        if (Auth::guard('admin')->check()) {
            $chat->is_active = true;
            $chat->admin_id = Auth::guard('admin')->id(); // Assign to current admin
            $chat->save();

            return redirect()->route('admin.chat.view', $chat)->with('success', 'Chat reopened successfully!');
        }
        return redirect()->route('admin.chats');
    }

    // Check typing status
    public function checkTyping(ChatRoom $chat)
    {
        $typingStatus = [];

        // Check user typing
        $userTypingKey = "chat_{$chat->id}_typing_user_{$chat->user_id}";
        $typingStatus['user_typing'] = cache()->has($userTypingKey);

        // Check admin typing
        if ($chat->admin_id) {
            $adminTypingKey = "chat_{$chat->id}_typing_admin_{$chat->admin_id}";
            $typingStatus['admin_typing'] = cache()->has($adminTypingKey);
        } else {
            $typingStatus['admin_typing'] = false;
        }

        return response()->json($typingStatus, 200, [], JSON_UNESCAPED_UNICODE);
    }

    // Get new messages
    public function getNewMessages(ChatRoom $chat, Request $request)
    {
        $lastMessageId = $request->get('last_message_id', 0);
        $includeAll = $request->get('include_all', false);
        $messages = app(\App\Services\ChatFileService::class)->getMessages($chat->id);

        if ($includeAll) {
            $result = array_map(function($msg) {
                return [
                    'id' => $msg['id'],
                    'sender_type' => $msg['sender_type'],
                    'sender_name' => $msg['sender_name'] ?? '',
                    'message' => $msg['message'],
                    'created_at' => $msg['created_at'],
                    'delivery_status' => $msg['delivery_status'] ?? 'sent',
                ];
            }, $messages);
        } else {
            $newMessages = array_filter($messages, function($msg) use ($lastMessageId) {
                return isset($msg['id']) && $msg['id'] > $lastMessageId;
            });
            $result = array_map(function($msg) {
                return [
                    'id' => $msg['id'],
                    'sender_type' => $msg['sender_type'],
                    'sender_name' => $msg['sender_name'] ?? '',
                    'message' => $msg['message'],
                    'created_at' => $msg['created_at'],
                    'delivery_status' => $msg['delivery_status'] ?? 'sent',
                ];
            }, $newMessages);
        }

        $lastId = !empty($result) ? end($result)['id'] : $lastMessageId;
        return response()->json([
            'success' => true,
            'messages' => array_values($result),
            'last_message_id' => $lastId
        ], 200, [], JSON_UNESCAPED_UNICODE);
    }

    // Mark messages as delivered
    public function markAsDelivered(ChatRoom $chat)
    {
        $messages = app(ChatFileService::class)->getMessages($chat->id);
        $updatedMessageIds = [];
        $isAdmin = Auth::guard('admin')->check() && $chat->admin_id == Auth::guard('admin')->id();
        $isUser = Auth::check() && $chat->user_id == Auth::id();
        $now = now()->toDateTimeString();
        foreach ($messages as &$msg) {
            if ($isUser && $msg['sender_type'] === 'admin' && $msg['delivery_status'] === 'sent') {
                $msg['delivery_status'] = 'delivered';
                $msg['updated_at'] = $now;
                $updatedMessageIds[] = $msg['id'];
            } elseif ($isAdmin && $msg['sender_type'] === 'user' && $msg['delivery_status'] === 'sent') {
                $msg['delivery_status'] = 'delivered';
                $msg['updated_at'] = $now;
                $updatedMessageIds[] = $msg['id'];
            }
        }
        unset($msg);
        app(ChatFileService::class)->updateMessages($chat->id, $messages);
        return response()->json([
            'success' => true,
            'updated_message_ids' => $updatedMessageIds
        ], 200, [], JSON_UNESCAPED_UNICODE);
    }

    // تعداد پیام خوانده‌نشده برای هر چت ادمین از فایل
    public function adminUnreadMessages()
    {
        // همه چت‌های فعال را بگیر
        $chats = \App\Models\ChatRoom::where('is_active', true)->get();
        $unreadCounts = [];
        $totalUnread = 0;
        foreach ($chats as $chat) {
            $messages = app(\App\Services\ChatFileService::class)->getMessages($chat->id);
            $unread = collect($messages)
                ->filter(function($msg) {
                    return isset($msg['sender_type'], $msg['is_read'])
                        && $msg['sender_type'] === 'user'
                        && ($msg['is_read'] === false || $msg['is_read'] === 0 || $msg['is_read'] === '0' || $msg['is_read'] === 'false');
                })
                ->count();
            $unreadCounts[$chat->id] = $unread;
            $totalUnread += $unread;
        }
        return response()->json([
            'success' => true,
            'unread_counts' => $unreadCounts,
            'total_unread' => $totalUnread,
            'unread_count' => $totalUnread,
        ], 200, [], JSON_UNESCAPED_UNICODE);
    }

    // New: AJAX endpoint to refresh chat lock
    public function refreshLock(ChatRoom $chat)
    {
        if ($chat->is_active) {
            $lockKey = "chat_{$chat->id}_lock";
            $currentAdminId = Auth::guard('admin')->id();
            $lockedBy = cache()->get($lockKey);
            if ($lockedBy && $lockedBy != $currentAdminId) {
                return response()->json(['success' => false, 'locked' => true]);
            }
            cache()->put($lockKey, $currentAdminId, 20);
            return response()->json([
                'success' => true,
                'locked' => false
            ], 200, [], JSON_UNESCAPED_UNICODE);
        }
        return response()->json([
            'success' => true,
            'locked' => false
        ], 200, [], JSON_UNESCAPED_UNICODE);
    }
}

