<?php

namespace App\Http\Controllers;

use App\Models\Cart;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Setting;
use App\Services\PaymentService;
use App\Services\InvoiceService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Http;

class CheckoutController extends Controller
{
    protected $paymentService;
    protected $invoiceService;

    public function __construct(PaymentService $paymentService, InvoiceService $invoiceService)
    {
        $this->paymentService = $paymentService;
        $this->invoiceService = $invoiceService;
    }

    public function index()
    {
        $cartItems = Cart::with('product')
            ->where('user_id', Auth::id())
            ->get();

        if ($cartItems->isEmpty()) {
            return redirect()->route('cart.index')
                ->with('error', 'Your cart is empty.');
        }

        $total = $cartItems->sum(function ($item) {
            return $item->quantity * ($item->total_price ?? $item->product->price);
        });

        // Calculate total shipping cost
        $shippingTotal = $cartItems->sum(function ($item) {
            return $item->product->shipping_cost ? ($item->product->shipping_cost * $item->quantity) : 0;
        });

        // Calculate tax
        $tax = $total * Setting::getTaxRate();
        $totalWithTax = $total + $shippingTotal + $tax;

        // Payment gateways
        $all_gateways = [
            'paypal' => 'PayPal',
            'stripe' => 'Stripe',
            'nowpayments' => 'NOWPayments',
            'bitpay' => 'BitPay',
            'coingate' => 'Coingate',
            'payoneer' => 'Payoneer',
            'wise' => 'Wise',
        ];

        // Ensure all gateways are enabled by default
        $enabled_gateways = Setting::getValue('enabled_gateways', array_keys($all_gateways));
        if (empty($enabled_gateways)) {
            Setting::setValue('enabled_gateways', array_keys($all_gateways), 'json');
            $enabled_gateways = array_keys($all_gateways);
        }

        $nonCryptoGateways = [];
        $cryptoGateways = [];
        foreach ($all_gateways as $key => $label) {
            if (in_array($key, $enabled_gateways)) {
                if (in_array($key, ['paypal', 'stripe', 'payoneer', 'wise'])) {
                    $nonCryptoGateways[$key] = $label;
                } else {
                    $cryptoGateways[$key] = $label;
                }
            }
        }

        $gateway_tokens = [];
        $gateway_fields = [];
        foreach ($all_gateways as $key => $label) {
            $gateway_tokens[$key] = Setting::getValue('gateway_token_' . $key, '');
        }
        $gateway_fields['paypal'] = [
            'client_id' => Setting::getValue('gateway_paypal_client_id', ''),
            'client_secret' => Setting::getValue('gateway_paypal_client_secret', ''),
            'mode' => Setting::getValue('gateway_paypal_mode', 'sandbox'),
        ];
        $gateway_fields['stripe'] = [
            'publishable_key' => Setting::getValue('gateway_stripe_publishable_key', ''),
            'secret_key' => Setting::getValue('gateway_stripe_secret_key', ''),
            'webhook_secret' => Setting::getValue('gateway_stripe_webhook_secret', ''),
        ];
        $gateway_fields['nowpayments'] = [
            'api_key' => Setting::getValue('gateway_nowpayments_api_key', ''),
            'ipn_secret' => Setting::getValue('gateway_nowpayments_ipn_secret', ''),
        ];
        $gateway_fields['bitpay'] = [
            'api_token' => Setting::getValue('gateway_bitpay_api_token', ''),
            'pairing_code' => Setting::getValue('gateway_bitpay_pairing_code', ''),
            'environment' => Setting::getValue('gateway_bitpay_environment', 'test'),
        ];
        $gateway_fields['coingate'] = [
            'api_key' => Setting::getValue('gateway_coingate_api_key', ''),
            'api_secret' => Setting::getValue('gateway_coingate_api_secret', ''),
            'environment' => Setting::getValue('gateway_coingate_environment', 'sandbox'),
        ];
        $gateway_fields['payoneer'] = [
            'client_id' => Setting::getValue('gateway_payoneer_client_id', ''),
            'client_secret' => Setting::getValue('gateway_payoneer_client_secret', ''),
            'partner_id' => Setting::getValue('gateway_payoneer_partner_id', ''),
            'api_url' => Setting::getValue('gateway_payoneer_api_url', ''),
        ];
        $gateway_fields['wise'] = [
            'api_token' => Setting::getValue('gateway_wise_api_token', ''),
            'profile_id' => Setting::getValue('gateway_wise_profile_id', ''),
            'environment' => Setting::getValue('gateway_wise_environment', 'sandbox'),
        ];

        $taxRate = Setting::getTaxRate();
        return view('checkout.index', compact('cartItems', 'total', 'tax', 'totalWithTax', 'shippingTotal', 'enabled_gateways', 'gateway_tokens', 'gateway_fields', 'all_gateways', 'nonCryptoGateways', 'cryptoGateways', 'taxRate'));
    }

    public function process(Request $request)
    {
        try {
            // Get current cart items
            $cartItems = Cart::with('product')
                ->where('user_id', Auth::id())
                ->get();

            if ($cartItems->isEmpty()) {
                return redirect()->route('cart.index')
                    ->with('error', 'Your cart is empty.');
            }

            // Get all enabled gateways for validation
            $all_gateways = [
                'paypal' => 'PayPal',
                'stripe' => 'Stripe',
                'nowpayments' => 'NOWPayments',
                'bitpay' => 'BitPay',
                'coingate' => 'Coingate',
                'payoneer' => 'Payoneer',
                'wise' => 'Wise',
            ];
            $enabled_gateways = Setting::getValue('enabled_gateways', array_keys($all_gateways));
            $cryptoGateways = [];
            foreach ($all_gateways as $key => $label) {
                if (in_array($key, $enabled_gateways) && in_array($key, ['nowpayments', 'bitpay', 'coingate'])) {
                    $cryptoGateways[] = $key;
                }
            }
            $validMethods = $enabled_gateways;
            Log::info('Checkout payment debug', [
                'enabled_gateways' => $enabled_gateways,
                'payment_method' => $request->input('payment_method'),
                'all_request' => $request->all()
            ]);
            $validator = Validator::make($request->all(), [
                'first_name' => 'required|string|max:255',
                'last_name' => 'required|string|max:255',
                'email' => 'required|email|max:255',
                'phone' => 'required|string|max:20',
                'shipping_address' => 'required|string',
                'country' => 'required|string|max:255',
                'city' => 'required|string|max:255',
                'payment_method' => 'required|in:' . implode(',', $validMethods),
                'order_notes' => 'nullable|string',
                'terms' => 'required|accepted',
            ]);

            if ($validator->fails()) {
                return redirect()->back()
                    ->withErrors($validator)
                    ->withInput()
                    ->with('error', 'Please fix the validation errors below.');
            }

            // Calculate total including additional prices with debugging
            $subtotal = 0;
            $shippingTotal = 0;
            foreach ($cartItems as $item) {
                $itemTotal = $item->quantity * ($item->total_price ?? $item->product->price);
                $subtotal += $itemTotal;
                $shippingTotal += $item->product->shipping_cost ? ($item->product->shipping_cost * $item->quantity) : 0;
            }

            // Calculate tax
            $tax = $subtotal * Setting::getTaxRate();
            $total = $subtotal + $shippingTotal + $tax;

            // Create order
            $order = Order::create([
                'user_id' => Auth::id(),
                'total_amount' => number_format($total, 2, '.', ''),
                'status' => 'pending',
                'shipping_address' => $request->input('shipping_address'),
                'country' => $request->input('country'),
                'city' => $request->input('city'),
                'payment_method' => $request->input('payment_method'),
                'first_name' => $request->input('first_name'),
                'last_name' => $request->input('last_name'),
                'email' => $request->input('email'),
                'phone' => $request->input('phone'),
                'order_notes' => $request->input('order_notes'),
            ]);

            // Create order items with final price from cart
            foreach ($cartItems as $item) {
                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $item->product_id,
                    'quantity' => $item->quantity,
                    'price' => $item->total_price ?? $item->product->price,
                    'dynamic_selections' => $item->dynamic_selections,
                    'additional_price' => $item->additional_price,
                ]);
            }

            // Process payment based on selected method
            if ($request->input('payment_method') === 'paypal') {
                $paymentResult = $this->paymentService->createPayPalPayment($order);
            } elseif (in_array($request->input('payment_method'), $cryptoGateways)) {
                $paymentResult = $this->paymentService->createCryptoPayment($order, $request);
                if (!$paymentResult['success']) {
                    $errorMsg = 'Crypto payment is not available. ';
                    if (isset($paymentResult['message'])) {
                        $errorMsg .= $paymentResult['message'];
                    } elseif (isset($paymentResult['response']['message'])) {
                        $errorMsg .= $paymentResult['response']['message'];
                    } elseif (isset($paymentResult['response']['code']) && $paymentResult['response']['code'] === 'BAD_REQUEST') {
                        $errorMsg .= 'Your order amount exceeds the maximum allowed for crypto payments.';
                    } else {
                        $errorMsg .= 'Please use another payment method or try again later.';
                    }
                    return redirect()->back()
                        ->with('error', $errorMsg)
                        ->withInput();
                }
            }

            // Store payment ID in session
            session(['payment_id' => $paymentResult['payment_id'] ?? null]);

            // Redirect based on payment method
            if ($request->input('payment_method') === 'paypal') {
                return redirect($paymentResult['redirect_url'])
                    ->with('success', 'Your order has been placed successfully! Redirecting to payment...');
            } else {
                // For crypto payments, show the payment page
                return view('checkout.crypto-payment', [
                    'payment' => $paymentResult,
                    'order' => $order
                ])->with('success', 'Your order has been placed successfully! Please complete the payment.');
            }
        } catch (\Exception $e) {
            Log::error('Checkout error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => Auth::id()
            ]);

            return redirect()->back()
                ->with('error', 'An error occurred while processing your order. Please try again.')
                ->withInput();
        }
    }

    public function success(Request $request)
    {
        try {
            $order = Order::where('user_id', Auth::id())
                ->latest()
                ->first();

            if (!$order) {
                return redirect()->route('home')
                    ->with('error', 'Order not found.');
            }

            // Verify payment status
            $paymentResult = $this->paymentService->verifyPayment($order);

            if ($paymentResult['success']) {
                // Update order status
                $order->update(['status' => 'paid']);

                // Send invoice email
                $this->invoiceService->sendInvoiceEmail($order);

                // Clear cart
                Cart::where('user_id', Auth::id())->delete();

                // Clear session
                session()->forget(['payment_id', 'checkout_processed']);

                return redirect()->route('orders.show', $order)
                    ->with('success', 'Payment successful! Thank you for your purchase.');
            } else {
                $order->update(['status' => 'failed']);
                return redirect()->route('checkout.index')
                    ->with('error', 'Payment verification failed. Please try again.');
            }
        } catch (\Exception $e) {
            Log::error('Payment success error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->route('checkout.index')
                ->with('error', 'An error occurred while verifying your payment. Please contact support.');
        }
    }

    public function cancel()
    {
        try {
            $order = Order::where('user_id', Auth::id())
                ->latest()
                ->first();

            if ($order) {
                $order->update(['status' => 'cancelled']);
            }

            // Clear session
            session()->forget(['payment_id', 'checkout_processed']);

            return redirect()->route('cart.index')
                ->with('error', 'Payment was cancelled.');
        } catch (\Exception $e) {
            Log::error('Payment cancellation error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->route('cart.index')
                ->with('error', 'An error occurred while cancelling your payment. Please try again.');
        }
    }

    public function availableCryptos(Request $request)
    {
        $amount = $request->input('amount');
        $currencies = ['BTC', 'ETH', 'USDT', 'DOGE']; // Add more as needed
        $available = [];
        $apiKey = config('services.nowpayments.api_key') ?? Setting::getValue('gateway_nowpayments_api_key', '');
        foreach ($currencies as $currency) {
            $response = \Illuminate\Support\Facades\Http::withHeaders([
                'x-api-key' => $apiKey
            ])->get('https://api.nowpayments.io/v1/estimate', [
                'amount' => number_format($amount, 2, '.', ''),
                'currency_from' => 'usd',
                'currency_to' => strtolower($currency)
            ]);
            if ($response->successful() && isset($response['estimated_amount'])) {
                $available[] = $currency;
            }
        }
        return response()->json($available);
    }
}
