<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class HomeController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::query()
            ->where('is_approved', true)
            ->with(['category']);

        $searchTerm = $request->get('search');
        $categoryId = $request->get('category');
        $sortBy = $request->get('sort', 'newest');
        $category = null;

        // Search functionality
        if ($searchTerm && !empty($searchTerm)) {
            $query->where(function($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('description', 'like', "%{$searchTerm}%")
                  ->orWhere('meta_keywords', 'like', "%{$searchTerm}%");
            });
        }

        // Category filter
        if ($categoryId && !empty($categoryId)) {
            $query->where('category_id', $categoryId);
            $category = Category::find($categoryId);

            // Auto-generate SEO data for category if needed
            if ($category && (!$category->meta_title || !$category->meta_description)) {
                $category->autoGenerateSeoFields();
            }
        }

        // Sorting
        switch ($sortBy) {
            case 'newest':
                $query->latest();
                break;
            case 'price_asc':
                $query->orderBy('price', 'asc');
                break;
            case 'price_desc':
                $query->orderBy('price', 'desc');
                break;
            case 'name':
                $query->orderBy('name', 'asc');
                break;
            default:
                $query->latest();
        }

        $products = $query->paginate(12)->withQueryString();
        $featuredProducts = Product::where('is_approved', true)
            ->where('is_featured', true)
            ->with(['category'])
            ->orderBy('price', 'desc')
            ->take(8)
            ->get();

        // Get low stock products (stock <= 10)
        $lowStockProducts = Product::where('is_approved', true)
            ->where('stock', '<=', 10)
            ->where('stock', '>', 0)
            ->with(['category'])
            ->orderBy('stock', 'asc')
            ->take(8)
            ->get();

        $categories = Category::all();

        // Generate SEO data
        $seoData = $this->generateSeoData($searchTerm, $category, $sortBy, $products, $request, $categoryId);

        return view('home', compact('products', 'featuredProducts', 'categories', 'lowStockProducts', 'seoData'));
    }

    private function generateSeoData($searchTerm, $category, $sortBy, $products, $request, $categoryId)
    {
        $seoData = [];

        // Generate title
        if ($searchTerm && $category) {
            $seoData['title'] = "Search results for '{$searchTerm}' in {$category->name} - " . config('app.name');
        } elseif ($searchTerm) {
            $seoData['title'] = "Search results for '{$searchTerm}' - " . config('app.name');
        } elseif ($category) {
            // Use category's meta_title from database if available
            $seoData['title'] = $category->meta_title ?: "{$category->name} Products - " . config('app.name');
        } else {
            $seoData['title'] = "Luxury Products - " . config('app.name');
        }

        // Generate description
        if ($searchTerm && $category) {
            $seoData['description'] = "Find the best {$category->name} products matching '{$searchTerm}'. Browse through {$products->total()} luxury items with secure checkout and fast delivery.";
        } elseif ($searchTerm) {
            $seoData['description'] = "Search results for '{$searchTerm}'. Discover {$products->total()} luxury products with premium quality and exclusive designs.";
        } elseif ($category) {
            // Use category's meta_description from database if available
            $seoData['description'] = $category->meta_description ?: "Explore our premium collection of {$category->name} products. {$products->total()} luxury items available with secure checkout and fast delivery.";
        } else {
            $seoData['description'] = "Discover authentic luxury brands and exclusive products at " . config('app.name') . ". Shop the world's best designer items with fast delivery and secure checkout.";
        }

        // Generate keywords
        $keywords = [];
        if ($searchTerm) {
            $keywords[] = $searchTerm;
        }
        if ($category) {
            $keywords[] = $category->name;
            // Use category meta_keywords from database if available
            if ($category->meta_keywords) {
                $categoryKeywords = explode(',', $category->meta_keywords);
                $keywords = array_merge($keywords, array_map('trim', $categoryKeywords));
            } else {
                // Auto-generate keywords for category if not set
                $category->autoGenerateSeoFields();
                if ($category->meta_keywords) {
                    $categoryKeywords = explode(',', $category->meta_keywords);
                    $keywords = array_merge($keywords, array_map('trim', $categoryKeywords));
                }
            }
        }
        $keywords = array_merge($keywords, ['luxury', 'premium', 'designer', 'authentic', config('app.name')]);
        $seoData['keywords'] = implode(', ', array_unique($keywords));
        $seoData['meta_keywords'] = $seoData['keywords'];

        // Generate canonical URL
        $seoData['canonical'] = url()->current();
        if ($request->getQueryString()) {
            $seoData['canonical'] = url()->current() . '?' . $request->getQueryString();
        }

        // Generate Open Graph data
        $seoData['og_title'] = $seoData['title'];
        $seoData['og_description'] = $seoData['description'];
        $seoData['og_type'] = 'website';
        $seoData['og_url'] = $seoData['canonical'];
        $seoData['og_image'] = asset('logo/logo.png');

        // Generate Twitter Card data
        $seoData['twitter_card'] = 'summary_large_image';
        $seoData['twitter_title'] = $seoData['title'];
        $seoData['twitter_description'] = $seoData['description'];
        $seoData['twitter_image'] = $seoData['og_image'];

        // Generate breadcrumbs
        $seoData['breadcrumbs'] = [
            ['url' => route('home'), 'title' => 'Home'],
        ];

        if ($category) {
            $seoData['breadcrumbs'][] = [
                'url' => route('home', ['category' => $category->id]),
                'title' => $category->name
            ];
        }

        if ($searchTerm) {
            $seoData['breadcrumbs'][] = [
                'url' => route('home', array_filter(['search' => $searchTerm, 'category' => $categoryId])),
                'title' => "Search: {$searchTerm}"
            ];
        }

        // Generate structured data
        $seoData['structured_data'] = $this->generateStructuredData($searchTerm, $category, $products);

        return $seoData;
    }

    private function generateStructuredData($searchTerm, $category, $products)
    {
        $structuredData = [
            '@context' => 'https://schema.org',
            '@type' => 'WebSite',
            'name' => config('app.name'),
            'url' => url('/'),
            'potentialAction' => [
                '@type' => 'SearchAction',
                'target' => url('/?search={search_term_string}'),
                'query-input' => 'required name=search_term_string'
            ]
        ];

        if ($category) {
            $structuredData['@type'] = 'CollectionPage';
            $structuredData['name'] = $category->name;
            $structuredData['description'] = $category->meta_description ?: "Collection of {$category->name} products";

            if ($category->schema_markup && is_array($category->schema_markup)) {
                $structuredData = array_merge($structuredData, $category->schema_markup);
            }
        }

        return $structuredData;
    }
}
