<?php

namespace App\Mail;

use App\Models\Product;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Mail\Mailable;
use Illuminate\Mail\Mailables\Content;
use Illuminate\Mail\Mailables\Envelope;
use Illuminate\Queue\SerializesModels;

class ProductNewsletter extends Mailable
{
    use Queueable, SerializesModels;

    public $product;
    public $type;
    public $firstImage;
    public $productUrl;

    /**
     * Create a new message instance.
     */
    public function __construct(Product $product, $type = 'new')
    {
        $this->product = $product;
        $this->type = $type;
        $images = is_string($product->image) ? json_decode($product->image, true) : $product->image;
        $this->firstImage = !empty($images) ? (is_array($images) ? $images[0] : $images) : null;
        $this->productUrl = url('/products/' . $product->slug);
    }

    /**
     * Get the message envelope.
     */
    public function envelope(): Envelope
    {
        return new Envelope(
            subject: 'Product Newsletter',
        );
    }

    /**
     * Get the message content definition.
     */
    public function content(): Content
    {
        return new Content(
            markdown: 'emails.newsletter.product',
        );
    }

    /**
     * Get the attachments for the message.
     *
     * @return array<int, \Illuminate\Mail\Mailables\Attachment>
     */
    public function attachments(): array
    {
        return [];
    }

    /**
     * Build the message.
     */
    public function build()
    {
        $subject = $this->type === 'new'
            ? 'New Product Added: ' . $this->product->name
            : 'Product Updated: ' . $this->product->name;

        // Increment emails_sent for each recipient
        if ($this->to) {
            $emails = is_array($this->to) ? $this->to : [$this->to];
            foreach ($emails as $email) {
                $subscriber = \App\Models\NewsletterSubscriber::where('email', $email)->first();
                if ($subscriber) {
                    $subscriber->increment('emails_sent');
                }
            }
        }

        return $this->subject($subject)
            ->markdown('emails.newsletter.product')
            ->with([
                'product' => $this->product,
                'firstImage' => $this->firstImage,
                'productUrl' => $this->productUrl,
                'type' => $this->type,
            ]);
    }
}
