<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Category extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'description',
        'image',
        'meta_title',
        'meta_description',
        'meta_keywords',
        'og_title',
        'og_description',
        'og_image',
        'twitter_title',
        'twitter_description',
        'twitter_image',
        'schema_markup',
        'seo_url',
        'canonical_url',
        'seo_indexable',
        'seo_priority',
        'category_type',
        'sort_order',
        'show_in_menu',
        'show_in_sitemap',
        'product_count',
        'avg_price'
    ];

    protected $casts = [
        'schema_markup' => 'json',
        'seo_indexable' => 'boolean',
        'show_in_menu' => 'boolean',
        'show_in_sitemap' => 'boolean',
        'product_count' => 'integer',
        'avg_price' => 'float'
    ];

    // SEO Accessors
    public function getSeoTitleAttribute()
    {
        return $this->meta_title ?: $this->name . ' - ' . config('app.name');
    }

    public function getSeoDescriptionAttribute()
    {
        return $this->meta_description ?: Str::limit(strip_tags($this->description), 160);
    }

    public function getSeoKeywordsAttribute()
    {
        if ($this->meta_keywords) {
            return $this->meta_keywords;
        }

        $keywords = [$this->name, 'category', 'products'];
        return implode(', ', array_filter($keywords));
    }

    public function getOgTitleAttribute($value)
    {
        return $value ?: $this->seo_title;
    }

    public function getOgDescriptionAttribute($value)
    {
        return $value ?: $this->seo_description;
    }

    public function getOgImageAttribute($value)
    {
        return $value ?: $this->image;
    }

    public function getTwitterTitleAttribute($value)
    {
        return $value ?: $this->seo_title;
    }

    public function getTwitterDescriptionAttribute($value)
    {
        return $value ?: $this->seo_description;
    }

    public function getTwitterImageAttribute($value)
    {
        return $value ?: $this->image;
    }

    public function getCanonicalUrlAttribute($value)
    {
        return $value ?: route('products.index', ['category' => $this->id]);
    }

    public function getSeoUrlAttribute($value)
    {
        return $value ?: $this->slug;
    }

    // Generate Schema.org markup for category
    public function generateSchemaMarkup()
    {
        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'CollectionPage',
            'name' => $this->name,
            'description' => $this->description,
            'url' => $this->canonical_url,
            'mainEntity' => [
                '@type' => 'ItemList',
                'numberOfItems' => $this->product_count,
                'itemListElement' => []
            ]
        ];

        if ($this->avg_price) {
            $schema['offers'] = [
                '@type' => 'AggregateOffer',
                'priceCurrency' => 'USD',
                'lowPrice' => $this->avg_price * 0.8, // Estimate
                'highPrice' => $this->avg_price * 1.2, // Estimate
                'offerCount' => $this->product_count
            ];
        }

        return $schema;
    }

    // Auto-generate SEO fields
    public function autoGenerateSeoFields()
    {
        if (!$this->meta_title) {
            $this->meta_title = $this->name . ' - ' . config('app.name');
        }

        if (!$this->meta_description) {
            $this->meta_description = Str::limit(strip_tags($this->description), 160);
        }

        if (!$this->meta_keywords) {
            $keywords = [$this->name, 'category', 'products', config('app.name')];
            $this->meta_keywords = implode(', ', array_filter($keywords));
        }

        if (!$this->schema_markup) {
            $this->schema_markup = $this->generateSchemaMarkup();
        }

        $this->save();
    }

    // Update product count and average price
    public function updateProductStats()
    {
        $products = $this->products()->where('is_approved', true);
        $this->product_count = $products->count();
        $this->avg_price = $products->avg('price');
        $this->save();
    }

    // Scope for SEO indexable categories
    public function scopeIndexable($query)
    {
        return $query->where('seo_indexable', true);
    }

    // Scope for menu categories
    public function scopeMenuCategories($query)
    {
        return $query->where('show_in_menu', true)->orderBy('sort_order');
    }

    // Scope for sitemap categories
    public function scopeSitemapCategories($query)
    {
        return $query->where('show_in_sitemap', true);
    }

    public function products()
    {
        return $this->hasMany(Product::class);
    }

    // Boot method to auto-generate SEO fields
    protected static function boot()
    {
        parent::boot();

        static::saved(function ($category) {
            if (!$category->meta_title || !$category->meta_description) {
                $category->autoGenerateSeoFields();
            }
        });
    }
}
