<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Product extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'description',
        'meta_title',
        'meta_description',
        'meta_keywords',
        'og_title',
        'og_description',
        'og_image',
        'twitter_title',
        'twitter_description',
        'twitter_image',
        'schema_markup',
        'seo_url',
        'canonical_url',
        'seo_indexable',
        'seo_priority',
        'image_alt_texts',
        'brand',
        'model',
        'sku',
        'gtin',
        'mpn',
        'rating_value',
        'rating_count',
        'price_range_min',
        'price_range_max',
        'availability',
        'condition',
        'price',
        'shipping_cost',
        'stock',
        'category_id',
        'admin_id',
        'dynamic_fields',
        'is_featured',
        'discount_percentage',
        'is_approved',
        'image',
        'user_id'
    ];

    protected $casts = [
        'price' => 'float',
        'shipping_cost' => 'float',
        'stock' => 'integer',
        'is_approved' => 'boolean',
        'is_featured' => 'boolean',
        'dynamic_fields' => 'json',
        'image' => 'array',
        'schema_markup' => 'json',
        'image_alt_texts' => 'json',
        'seo_indexable' => 'boolean',
        'rating_value' => 'float',
        'rating_count' => 'integer',
        'price_range_min' => 'float',
        'price_range_max' => 'float'
    ];

    // SEO Accessors
    public function getSeoTitleAttribute()
    {
        return $this->meta_title ?: $this->name;
    }

    public function getSeoDescriptionAttribute()
    {
        return $this->meta_description ?: Str::limit(strip_tags($this->description), 160);
    }

    public function getSeoKeywordsAttribute()
    {
        if ($this->meta_keywords) {
            return $this->meta_keywords;
        }

        $keywords = [$this->name, $this->brand, $this->category->name ?? ''];
        if ($this->dynamic_fields) {
            $keywords = array_merge($keywords, array_values($this->dynamic_fields));
        }

        return implode(', ', array_filter($keywords));
    }

    public function getOgTitleAttribute($value)
    {
        return $value ?: $this->seo_title;
    }

    public function getOgDescriptionAttribute($value)
    {
        return $value ?: $this->seo_description;
    }

    public function getOgImageAttribute($value)
    {
        return $value ?: $this->first_image;
    }

    public function getTwitterTitleAttribute($value)
    {
        return $value ?: $this->seo_title;
    }

    public function getTwitterDescriptionAttribute($value)
    {
        return $value ?: $this->seo_description;
    }

    public function getTwitterImageAttribute($value)
    {
        return $value ?: $this->first_image;
    }

    public function getCanonicalUrlAttribute($value)
    {
        return $value ?: route('products.show', $this);
    }

    public function getSeoUrlAttribute($value)
    {
        return $value ?: $this->slug;
    }

    // Auto-generate image alt texts
    public function generateImageAltTexts()
    {
        if (!$this->image) {
            return [];
        }

        $images = is_array($this->image) ? $this->image : json_decode($this->image, true);
        $altTexts = [];

        foreach ($images as $index => $image) {
            $altTexts[] = $this->name . ($index > 0 ? ' - Image ' . ($index + 1) : '') . ' - ' . config('app.name');
        }

        return $altTexts;
    }

    // Generate Schema.org markup
    public function generateSchemaMarkup()
    {
        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'Product',
            'name' => $this->name,
            'description' => $this->description,
            'brand' => [
                '@type' => 'Brand',
                'name' => $this->brand ?: config('app.name')
            ],
            'sku' => $this->sku,
            'mpn' => $this->mpn,
            'gtin' => $this->gtin,
            'category' => $this->category->name ?? '',
            'condition' => $this->condition,
            'availability' => $this->availability === 'in_stock' ? 'https://schema.org/InStock' : 'https://schema.org/OutOfStock',
            'price' => $this->discounted_price,
            'priceCurrency' => 'USD',
            'image' => $this->images,
        ];

        if ($this->rating_value && $this->rating_count > 0) {
            $schema['aggregateRating'] = [
                '@type' => 'AggregateRating',
                'ratingValue' => $this->rating_value,
                'reviewCount' => $this->rating_count,
                'bestRating' => 5,
                'worstRating' => 1
            ];
        }

        return $schema;
    }

    // Update rating from reviews
    public function updateRatingFromReviews()
    {
        $reviews = $this->reviews()->where('is_approved', true)->get();

        if ($reviews->count() > 0) {
            $this->rating_value = $reviews->avg('rating');
            $this->rating_count = $reviews->count();
        } else {
            $this->rating_value = null;
            $this->rating_count = 0;
        }

        $this->save();
    }

    // Auto-generate SEO fields
    public function autoGenerateSeoFields()
    {
        if (!$this->meta_title) {
            $this->meta_title = $this->name . ' - ' . config('app.name');
        }

        if (!$this->meta_description) {
            $this->meta_description = Str::limit(strip_tags($this->description), 160);
        }

        if (!$this->meta_keywords) {
            $keywords = [$this->name, $this->brand, $this->category->name ?? ''];
            $this->meta_keywords = implode(', ', array_filter($keywords));
        }

        if (!$this->image_alt_texts) {
            $this->image_alt_texts = $this->generateImageAltTexts();
        }

        if (!$this->schema_markup) {
            $this->schema_markup = $this->generateSchemaMarkup();
        }

        $this->save();
    }

    // Relationships
    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    public function admin()
    {
        return $this->belongsTo(Admin::class);
    }

    public function reviews()
    {
        return $this->hasMany(Review::class);
    }

    public function cartItems()
    {
        return $this->hasMany(Cart::class);
    }

    public function orderItems()
    {
        return $this->hasMany(OrderItem::class);
    }

    public function getFirstImageAttribute()
    {
        if (!$this->image) {
            return null;
        }

        $images = is_array($this->image) ? $this->image : json_decode($this->image, true);
        return !empty($images) ? $images[0] : null;
    }

    public function getImagesAttribute()
    {
        if (is_string($this->image)) {
            $images = json_decode($this->image, true);
            return is_array($images) ? $images : [];
        }

        return is_array($this->image) ? $this->image : [];
    }

    public function getDiscountedPriceAttribute()
    {
        if (!$this->discount_percentage) {
            return $this->price;
        }

        $discount = $this->price * ($this->discount_percentage / 100);
        return $this->price - $discount;
    }

    public function hasDiscount()
    {
        return !is_null($this->discount_percentage) && $this->discount_percentage > 0;
    }

    protected function setDynamicFieldsAttribute($value)
    {
        if (is_array($value)) {
            $this->attributes['dynamic_fields'] = json_encode($value);
        } else {
            $this->attributes['dynamic_fields'] = $value;
        }
    }

    protected function getDynamicFieldsAttribute($value)
    {
        return $value ? json_decode($value, true) : null;
    }

    // Boot method to auto-generate SEO fields
    protected static function boot()
    {
        parent::boot();

        static::saved(function ($product) {
            if (!$product->meta_title || !$product->meta_description) {
                $product->autoGenerateSeoFields();
            }
        });
    }
}
