<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class Setting extends Model
{
    use HasFactory;

    protected $fillable = [
        'key',
        'value',
        'type',
        'description'
    ];

    /**
     * Get a setting value by key
     */
    public static function getValue($key, $default = null)
    {
        $cacheKey = "setting_{$key}";

        return Cache::remember($cacheKey, 3600, function () use ($key, $default) {
            $setting = self::where('key', $key)->first();

            if (!$setting) {
                return $default;
            }

            switch ($setting->type) {
                case 'number':
                    return (float) $setting->value;
                case 'boolean':
                    return (bool) $setting->value;
                case 'json':
                    return json_decode($setting->value, true);
                default:
                    return $setting->value;
            }
        });
    }

    /**
     * Set a setting value
     */
    public static function setValue($key, $value, $type = 'string', $description = null)
    {
        $setting = self::updateOrCreate(
            ['key' => $key],
            [
                'value' => is_array($value) ? json_encode($value) : (string) $value,
                'type' => $type,
                'description' => $description
            ]
        );

        // Clear cache
        Cache::forget("setting_{$key}");

        return $setting;
    }

    /**
     * Get tax rate as decimal (e.g., 0.08 for 8%)
     */
    public static function getTaxRate()
    {
        return self::getValue('tax_rate', 0.08);
    }

    /**
     * Set tax rate
     */
    public static function setTaxRate($rate)
    {
        return self::setValue('tax_rate', $rate, 'number', 'Tax rate as decimal (e.g., 0.08 for 8%)');
    }
}
