<?php

namespace App\Services;

use App\Models\Order;
use App\Models\OrderItem;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class InvoiceService
{
    public function generateInvoice(Order $order)
    {
        try {
            // Get order with items and user
            $order->load(['items.product', 'user']);

            // Generate invoice number
            $invoiceNumber = 'INV-' . date('Y') . '-' . str_pad($order->id, 6, '0', STR_PAD_LEFT);

            // Calculate totals
            $subtotal = $order->items->sum(function ($item) {
                return $item->price * $item->quantity;
            });

            $tax = $subtotal * 0.08; // 8% tax
            $total = $subtotal + $tax;

            // Create invoice data
            $invoiceData = [
                'invoice_number' => $invoiceNumber,
                'order' => $order,
                'order_items' => $order->items,
                'subtotal' => $subtotal,
                'tax' => $tax,
                'total' => $total,
                'issue_date' => now()->format('Y-m-d'),
                'due_date' => now()->addDays(30)->format('Y-m-d'),
            ];

            // Store invoice data in session for PDF generation
            session(['invoice_data' => $invoiceData]);

            return $invoiceData;

        } catch (\Exception $e) {
            Log::error('Invoice generation error', [
                'message' => $e->getMessage(),
                'order_id' => $order->id
            ]);

            return null;
        }
    }

    public function sendInvoiceEmail(Order $order, $invoiceData = null)
    {
        try {
            if (!$invoiceData) {
                $invoiceData = $this->generateInvoice($order);
            }

            if (!$invoiceData) {
                throw new \Exception('Failed to generate invoice data');
            }

            // Send email
            Mail::send('emails.invoice', $invoiceData, function ($message) use ($order, $invoiceData) {
                $message->to($order->email, $order->first_name . ' ' . $order->last_name)
                        ->subject('Invoice for Order #' . $order->id . ' - ' . $invoiceData['invoice_number']);
            });

            Log::info('Invoice email sent successfully', [
                'order_id' => $order->id,
                'email' => $order->email,
                'invoice_number' => $invoiceData['invoice_number']
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error('Invoice email sending error', [
                'message' => $e->getMessage(),
                'order_id' => $order->id,
                'email' => $order->email ?? 'N/A'
            ]);

            return false;
        }
    }

    public function getInvoiceData($orderId)
    {
        $order = Order::with(['items.product', 'user'])->find($orderId);

        if (!$order) {
            return null;
        }

        return $this->generateInvoice($order);
    }
}
