<?php

namespace App\Services;

use App\Models\Order;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use PayPal\Api\Amount;
use PayPal\Api\Payer;
use PayPal\Api\Payment;
use PayPal\Api\RedirectUrls;
use PayPal\Api\Transaction;
use PayPal\Auth\OAuthTokenCredential;
use PayPal\Rest\ApiContext;
use Illuminate\Http\Request;

class PaymentService
{
    private $paypalClientId;
    private $paypalClientSecret;
    private $nowPaymentsApiKey;
    private $nowPaymentsPublicKey;
    private $paypalContext;
    private $paypalMode;

    public function __construct()
    {
        $this->paypalClientId = config('services.paypal.client_id');
        $this->paypalClientSecret = config('services.paypal.client_secret');
        $this->paypalMode = config('services.paypal.mode', 'sandbox');
        $this->nowPaymentsApiKey = config('services.nowpayments.api_key');
        $this->nowPaymentsPublicKey = config('services.nowpayments.public_key');

        $this->paypalContext = new ApiContext(
            new OAuthTokenCredential(
                $this->paypalClientId,
                $this->paypalClientSecret
            )
        );

        $this->paypalContext->setConfig([
            'mode' => $this->paypalMode,
            'log.LogEnabled' => true,
            'log.FileName' => storage_path('logs/paypal.log'),
            'log.LogLevel' => 'INFO'
        ]);
    }

    public function processPayment(Order $order, string $method)
    {
        try {
            Log::info('Starting payment process', [
                'order_id' => $order->id,
                'method' => $method,
                'amount' => $order->total_amount
            ]);

            if ($method === 'paypal') {
                return $this->processPayPalPayment($order);
            } elseif ($method === 'crypto') {
                return $this->processCryptoPayment($order);
            }

            Log::error('Invalid payment method', [
                'order_id' => $order->id,
                'method' => $method
            ]);

            return [
                'success' => false,
                'message' => 'Invalid payment method selected'
            ];
        } catch (\Exception $e) {
            Log::error('Payment processing error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $order->id,
                'method' => $method,
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);

            return [
                'success' => false,
                'message' => 'An error occurred while processing your payment. Please try again.'
            ];
        }
    }

    private function processPayPalPayment(Order $order)
    {
        try {
            Log::info('Processing PayPal payment', [
                'order_id' => $order->id,
                'amount' => $order->total_amount
            ]);

            if (empty($this->paypalClientId) || empty($this->paypalClientSecret)) {
                Log::error('PayPal credentials missing');
                return [
                    'success' => false,
                    'message' => 'Payment configuration error: PayPal credentials are missing'
                ];
            }

            // Get PayPal access token
            $tokenResponse = Http::withBasicAuth($this->paypalClientId, $this->paypalClientSecret)
                ->asForm()
                ->post('https://api-m.paypal.com/v1/oauth2/token', [
                    'grant_type' => 'client_credentials'
                ]);

            if (!$tokenResponse->successful()) {
                Log::error('PayPal token request failed', [
                    'response' => $tokenResponse->json(),
                    'order_id' => $order->id
                ]);
                return [
                    'success' => false,
                    'message' => 'Failed to initialize PayPal payment'
                ];
            }

            $accessToken = $tokenResponse->json()['access_token'];

            // Create PayPal order
            $orderResponse = Http::withToken($accessToken)
                ->post('https://api-m.paypal.com/v2/checkout/orders', [
                    'intent' => 'CAPTURE',
                    'purchase_units' => [
                        [
                            'reference_id' => $order->id,
                            'amount' => [
                                'currency_code' => 'USD',
                                'value' => number_format($order->total_amount, 2, '.', '')
                            ]
                        ]
                    ],
                    'application_context' => [
                        'return_url' => route('payment.paypal.success'),
                        'cancel_url' => route('payment.paypal.cancel')
                    ]
                ]);

            if (!$orderResponse->successful()) {
                Log::error('PayPal order creation failed', [
                    'response' => $orderResponse->json(),
                    'order_id' => $order->id
                ]);
                return [
                    'success' => false,
                    'message' => 'Failed to create PayPal order'
                ];
            }

            $paypalOrder = $orderResponse->json();

            // Store PayPal order ID in session
            session(['paypal_order_id' => $paypalOrder['id']]);

            Log::info('PayPal order created successfully', [
                'order_id' => $order->id,
                'paypal_order_id' => $paypalOrder['id']
            ]);

            return [
                'success' => true,
                'redirect_url' => collect($paypalOrder['links'])
                    ->firstWhere('rel', 'approve')['href']
            ];
        } catch (\Exception $e) {
            Log::error('PayPal payment error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $order->id
            ]);
            return [
                'success' => false,
                'message' => 'An error occurred while processing PayPal payment'
            ];
        }
    }

    private function processCryptoPayment(Order $order)
    {
        try {
            Log::info('Processing crypto payment', [
                'order_id' => $order->id,
                'amount' => $order->total_amount
            ]);

            if (empty($this->nowPaymentsApiKey)) {
                Log::error('NowPayments API key is missing');
                return [
                    'success' => false,
                    'message' => 'Payment configuration error: API key is missing'
                ];
            }

            // First, get the estimated amount
            $estimateResponse = Http::withHeaders([
                'x-api-key' => $this->nowPaymentsApiKey
            ])->get('https://api.nowpayments.io/v1/estimate', [
                'amount' => number_format($order->total_amount, 2, '.', ''),
                'currency_from' => 'usd',
                'currency_to' => 'btc'
            ]);

            Log::info('NowPayments estimate response', [
                'status' => $estimateResponse->status(),
                'body' => $estimateResponse->json(),
                'order_id' => $order->id
            ]);

            if (!$estimateResponse->successful()) {
                Log::error('NowPayments estimate failed', [
                    'response' => $estimateResponse->json(),
                    'order_id' => $order->id,
                    'status' => $estimateResponse->status()
                ]);
                return [
                    'success' => false,
                    'message' => 'Failed to get payment estimate: ' . ($estimateResponse->json()['message'] ?? 'Unknown error')
                ];
            }

            $estimateData = $estimateResponse->json();
            $estimatedAmount = $estimateData['estimated_amount'];

            // Create the payment
            $response = Http::withHeaders([
                'x-api-key' => $this->nowPaymentsApiKey
            ])->post('https://api.nowpayments.io/v1/payment', [
                'price_amount' => number_format($order->total_amount, 2, '.', ''),
                'price_currency' => 'usd',
                'pay_amount' => $estimatedAmount,
                'pay_currency' => 'btc',
                'order_id' => $order->id,
                'order_description' => "Order #{$order->id}",
                'ipn_callback_url' => route('payment.crypto.callback'),
                'success_url' => route('checkout.success'),
                'cancel_url' => route('checkout.cancel'),
                'is_fixed_rate' => true,
                'is_fee_paid_by_user' => true
            ]);

            Log::info('NowPayments payment creation response', [
                'status' => $response->status(),
                'body' => $response->json(),
                'order_id' => $order->id
            ]);

            if (!$response->successful()) {
                Log::error('NowPayments payment creation failed', [
                    'response' => $response->json(),
                    'order_id' => $order->id,
                    'status' => $response->status()
                ]);
                return [
                    'success' => false,
                    'message' => 'Failed to create payment: ' . ($response->json()['message'] ?? 'Unknown error')
                ];
            }

            $paymentData = $response->json();

            // Store payment ID in session for verification
            session(['nowpayments_payment_id' => $paymentData['payment_id']]);

            // Return payment details for the frontend
            return [
                'success' => true,
                'payment_id' => $paymentData['payment_id'],
                'pay_address' => $paymentData['pay_address'],
                'pay_amount' => $paymentData['pay_amount'],
                'pay_currency' => $paymentData['pay_currency'],
                'expires_at' => $paymentData['expiration_estimate_date'],
                'status' => $paymentData['payment_status']
            ];
        } catch (\Exception $e) {
            Log::error('Crypto payment error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $order->id
            ]);
            return [
                'success' => false,
                'message' => 'An error occurred while processing crypto payment'
            ];
        }
    }

    public function verifyPayment(Order $order)
    {
        try {
            if ($order->payment_method === 'paypal') {
                return $this->verifyPayPalPayment($order);
            } elseif ($order->payment_method === 'crypto') {
                return $this->verifyCryptoPayment($order);
            }

            return [
                'success' => false,
                'message' => 'Invalid payment method'
            ];
        } catch (\Exception $e) {
            Log::error('Payment verification error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $order->id
            ]);

            return [
                'success' => false,
                'message' => 'An error occurred while verifying payment'
            ];
        }
    }

    private function verifyPayPalPayment(Order $order)
    {
        try {
            $paymentId = session('payment_id');
            if (!$paymentId) {
                return [
                    'success' => false,
                    'message' => 'Payment ID not found'
                ];
            }

            // Get PayPal access token
            $tokenResponse = Http::withBasicAuth($this->paypalClientId, $this->paypalClientSecret)
                ->asForm()
                ->post('https://api-m.paypal.com/v1/oauth2/token', [
                    'grant_type' => 'client_credentials'
                ]);

            if (!$tokenResponse->successful()) {
                return [
                    'success' => false,
                    'message' => 'Failed to get PayPal access token'
                ];
            }

            $accessToken = $tokenResponse->json()['access_token'];

            // Check payment status
            $response = Http::withToken($accessToken)
                ->get("https://api-m.paypal.com/v2/checkout/orders/{$paymentId}");

            if (!$response->successful()) {
                return [
                    'success' => false,
                    'message' => 'Failed to verify PayPal payment'
                ];
            }

            $paymentData = $response->json();
            return [
                'success' => $paymentData['status'] === 'COMPLETED',
                'message' => $paymentData['status'] === 'COMPLETED' ? 'Payment verified' : 'Payment not completed'
            ];
        } catch (\Exception $e) {
            Log::error('PayPal verification error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $order->id
            ]);

            return [
                'success' => false,
                'message' => 'An error occurred while verifying PayPal payment'
            ];
        }
    }

    private function verifyCryptoPayment(Order $order)
    {
        try {
            $paymentId = session('payment_id');
            if (!$paymentId) {
                return [
                    'success' => false,
                    'message' => 'Payment ID not found'
                ];
            }

            $response = Http::withHeaders([
                'x-api-key' => $this->nowPaymentsApiKey
            ])->get("https://api.nowpayments.io/v1/payment/{$paymentId}");

            if (!$response->successful()) {
                return [
                    'success' => false,
                    'message' => 'Failed to verify crypto payment'
                ];
            }

            $paymentData = $response->json();
            return [
                'success' => $paymentData['payment_status'] === 'confirmed',
                'message' => $paymentData['payment_status'] === 'confirmed' ? 'Payment verified' : 'Payment not confirmed'
            ];
        } catch (\Exception $e) {
            Log::error('Crypto verification error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $order->id
            ]);

            return [
                'success' => false,
                'message' => 'An error occurred while verifying crypto payment'
            ];
        }
    }

    public function createCryptoPayment(Order $order, Request $request)
    {
        try {
            // Get payment estimate
            $estimateResponse = Http::withHeaders([
                'x-api-key' => $this->nowPaymentsApiKey
            ])->get('https://api.nowpayments.io/v1/estimate', [
                'amount' => number_format($order->total_amount, 2, '.', ''),
                'currency_from' => 'usd',
                'currency_to' => $request->input('crypto_currency', 'btc')
            ]);

            Log::info('NowPayments estimate response', [
                'status' => $estimateResponse->status(),
                'body' => $estimateResponse->json(),
                'order_id' => $order->id
            ]);

            if (!$estimateResponse->successful()) {
                Log::error('NowPayments estimate failed', [
                    'response' => $estimateResponse->json(),
                    'order_id' => $order->id,
                    'status' => $estimateResponse->status()
                ]);
                return [
                    'success' => false,
                    'message' => 'Failed to get payment estimate: ' . ($estimateResponse->json()['message'] ?? 'Unknown error')
                ];
            }

            $estimateData = $estimateResponse->json();
            $estimatedAmount = $estimateData['estimated_amount'];

            // Create the payment
            $response = Http::withHeaders([
                'x-api-key' => $this->nowPaymentsApiKey
            ])->post('https://api.nowpayments.io/v1/payment', [
                'price_amount' => number_format($order->total_amount, 2, '.', ''),
                'price_currency' => 'usd',
                'pay_amount' => $estimatedAmount,
                'pay_currency' => $request->input('crypto_currency', 'btc'),
                'order_id' => $order->id,
                'order_description' => "Order #{$order->id}",
                'ipn_callback_url' => route('payment.crypto.callback'),
                'success_url' => route('checkout.success'),
                'cancel_url' => route('checkout.cancel'),
                'is_fixed_rate' => true,
                'is_fee_paid_by_user' => true
            ]);

            Log::info('NowPayments payment creation response', [
                'status' => $response->status(),
                'body' => $response->json(),
                'order_id' => $order->id
            ]);

            if (!$response->successful()) {
                Log::error('NowPayments payment creation failed', [
                    'response' => $response->json(),
                    'order_id' => $order->id,
                    'status' => $response->status()
                ]);
                return [
                    'success' => false,
                    'message' => 'Failed to create payment: ' . ($response->json()['message'] ?? 'Unknown error')
                ];
            }

            $paymentData = $response->json();

            // Store payment ID in session for verification
            session(['nowpayments_payment_id' => $paymentData['payment_id']]);

            // Return payment details for the frontend
            return [
                'success' => true,
                'payment_id' => $paymentData['payment_id'],
                'pay_address' => $paymentData['pay_address'],
                'pay_amount' => $paymentData['pay_amount'],
                'pay_currency' => $paymentData['pay_currency'],
                'expires_at' => $paymentData['expiration_estimate_date'],
                'status' => $paymentData['payment_status']
            ];
        } catch (\Exception $e) {
            Log::error('Crypto payment error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $order->id
            ]);
            return [
                'success' => false,
                'message' => 'An error occurred while processing crypto payment'
            ];
        }
    }

    public function createPayPalPayment(Order $order)
    {
        try {
            $apiContext = new ApiContext(
                new \PayPal\Auth\OAuthTokenCredential(
                    $this->paypalClientId,
                    $this->paypalClientSecret
                )
            );

            $apiContext->setConfig([
                'mode' => $this->paypalMode,
                'log.LogEnabled' => true,
                'log.FileName' => storage_path('logs/paypal.log'),
                'log.LogLevel' => 'INFO'
            ]);

            $payer = new \PayPal\Api\Payer();
            $payer->setPaymentMethod('paypal');

            $amount = new \PayPal\Api\Amount();
            $amount->setTotal(number_format($order->total_amount, 2, '.', ''));
            $amount->setCurrency('USD');

            $transaction = new \PayPal\Api\Transaction();
            $transaction->setAmount($amount);
            $transaction->setDescription("Order #{$order->id}");
            $transaction->setCustom($order->id);

            $redirectUrls = new \PayPal\Api\RedirectUrls();
            $redirectUrls->setReturnUrl(route('checkout.success'));
            $redirectUrls->setCancelUrl(route('checkout.cancel'));

            $payment = new \PayPal\Api\Payment();
            $payment->setIntent('sale')
                ->setPayer($payer)
                ->setTransactions([$transaction])
                ->setRedirectUrls($redirectUrls);

            $payment->create($apiContext);

            return [
                'success' => true,
                'redirect_url' => $payment->getApprovalLink()
            ];
        } catch (\Exception $e) {
            Log::error('PayPal payment error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $order->id
            ]);
            return [
                'success' => false,
                'message' => 'An error occurred while processing PayPal payment'
            ];
        }
    }
}
