<!DOCTYPE html>
<html lang="{{ str_replace('_', '-', app()->getLocale()) }}">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>Chat Support</title>

    <link rel="stylesheet" href="{{ asset('admin/css/global.css') }}">

    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
            background-color: #111827;
            color: #f3f4f6;
            line-height: 1.5;
        }

        .chat-container {
            height: 100vh;
            height: 100dvh;
            display: flex;
            flex-direction: column;
            background: #1f2937;
            overflow: hidden;
        }

        .chat-header {
            background: linear-gradient(135deg, #3b82f6, #8b5cf6);
            padding: 0.75rem 1rem;
            flex-shrink: 0;
            border-bottom: 1px solid #374151;
        }

        .chat-header-content {
            display: flex;
            align-items: center;
            justify-content: space-between;
        }

        .chat-title {
            font-size: 1rem;
            font-weight: 700;
            color: white;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            max-width: 60%;
        }

        .chat-status {
            display: flex;
            flex-direction: column;
            align-items: flex-end;
            gap: 0.25rem;
        }

        .status-badge {
            font-size: 0.75rem;
            padding: 0.25rem 0.5rem;
            border-radius: 9999px;
            white-space: nowrap;
            font-weight: 600;
        }

        .status-closed { background-color: #ef4444; color: white; }
        .status-assigned { background-color: #10b981; color: white; }
        .status-waiting { background-color: #f59e0b; color: white; }
        .status-unread { background-color: #ef4444; color: white; }

        .notice {
            background: #1e40af;
            border-left: 4px solid #3b82f6;
            color: #dbeafe;
            padding: 0.75rem 1rem;
            flex-shrink: 0;
        }

        .notice-content {
            display: flex;
            align-items: flex-start;
            gap: 0.75rem;
        }

        .notice-icon {
            width: 1rem;
            height: 1rem;
            flex-shrink: 0;
            margin-top: 0.125rem;
        }

        .notice-text {
            font-size: 0.875rem;
        }

        .status-indicator {
            background: #374151;
            padding: 0.5rem 1rem;
            border-bottom: 1px solid #4b5563;
            display: none;
            flex-shrink: 0;
        }

        .status-indicator.show {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .spinner {
            width: 1rem;
            height: 1rem;
            border: 2px solid #6b7280;
            border-top: 2px solid #3b82f6;
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        .status-text {
            font-size: 0.875rem;
            color: #d1d5db;
        }

        .chat-messages {
            flex: 1;
            overflow-y: auto;
            padding: 1rem;
            background: #111827;
            scroll-behavior: smooth;
        }

        .message-container {
            margin-bottom: 1rem;
            display: flex;
        }

        .message-container.user {
            justify-content: flex-start;
        }

        .message-container.admin {
            justify-content: flex-end;
        }

        .message-bubble {
            max-width: 80%;
            padding: 0.75rem 1rem;
            border-radius: 1rem;
            word-wrap: break-word;
        }

        .message-user {
            background: linear-gradient(135deg, #3b82f6, #1d4ed8);
            color: white;
        }

        .message-admin {
            background: linear-gradient(135deg, #10b981, #059669);
            color: white;
        }

        .message-sender {
            font-size: 0.75rem;
            font-weight: 600;
            margin-bottom: 0.25rem;
            opacity: 0.9;
        }

        .message-text {
            font-size: 0.875rem;
            line-height: 1.5;
            margin-bottom: 0.25rem;
        }

        .message-footer {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-top: 0.25rem;
        }

        .message-time {
            font-size: 0.75rem;
            opacity: 0.75;
        }

        .message-status {
            font-size: 0.75rem;
            margin-left: 0.5rem;
            font-weight: 700;
        }

        .status-sent { color: rgba(255, 255, 255, 0.6); }
        .status-delivered { color: #93c5fd; }
        .status-read { color: #86efac; }

        .typing-indicator {
            background: #374151;
            padding: 0.5rem 1rem;
            border-top: 1px solid #4b5563;
            display: none;
            flex-shrink: 0;
        }

        .typing-indicator.show {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .typing-content {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .typing-text {
            font-size: 0.875rem;
            color: #d1d5db;
        }

        .typing-dots {
            display: flex;
            gap: 0.25rem;
        }

        .typing-dot {
            width: 0.5rem;
            height: 0.5rem;
            background: #6b7280;
            border-radius: 50%;
            animation: typing 1.4s infinite;
        }

        .typing-dot:nth-child(2) { animation-delay: 0.2s; }
        .typing-dot:nth-child(3) { animation-delay: 0.4s; }

        @keyframes typing {
            0%, 60%, 100% { transform: translateY(0); }
            30% { transform: translateY(-0.5rem); }
        }

        .chat-input-section {
            border-top: 1px solid #374151;
            background: #1f2937;
            padding: 1rem;
            flex-shrink: 0;
        }

        .warning-notice {
            background: #92400e;
            border-left: 4px solid #f59e0b;
            color: #fef3c7;
            padding: 0.75rem;
            margin-bottom: 0.75rem;
            border-radius: 0.5rem;
        }

        .warning-content {
            display: flex;
            align-items: flex-start;
            gap: 0.75rem;
        }

        .warning-icon {
            width: 1rem;
            height: 1rem;
            flex-shrink: 0;
            margin-top: 0.125rem;
        }

        .warning-text {
            font-size: 0.875rem;
        }

        .chat-form {
            display: flex;
            gap: 0.5rem;
            align-items: stretch;
        }

        .message-input {
            flex: 1;
            border: 1px solid #4b5563;
            background: #374151;
            color: #f3f4f6;
            border-radius: 0.5rem;
            padding: 0.75rem 1rem;
            font-size: 0.875rem;
            outline: none;
            transition: all 0.3s ease;
        }

        .message-input:focus {
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }

        .message-input::placeholder {
            color: #9ca3af;
        }

        .btn {
            padding: 0.75rem 1rem;
            border-radius: 0.5rem;
            font-size: 0.875rem;
            font-weight: 600;
            border: none;
            cursor: pointer;
            transition: all 0.3s ease;
            outline: none;
        }

        .btn-primary {
            background: #3b82f6;
            color: white;
        }

        .btn-primary:hover {
            background: #2563eb;
        }

        .btn-secondary {
            background: #6b7280;
            color: white;
        }

        .btn-secondary:hover {
            background: #4b5563;
        }

        .btn-danger {
            background: #ef4444;
            color: white;
        }

        .btn-danger:hover {
            background: #dc2626;
        }

        .close-chat-section {
            border-top: 1px solid #374151;
            background: #1f2937;
            padding: 1rem;
            text-align: center;
            flex-shrink: 0;
        }

        /* Mobile Responsive */
        @media (max-width: 640px) {
            .chat-header {
                padding: 0.5rem 0.75rem;
            }

            .chat-title {
                font-size: 0.875rem;
                max-width: 50%;
            }

            .chat-status {
                align-items: flex-end;
            }

            .status-badge {
                font-size: 0.625rem;
                padding: 0.125rem 0.375rem;
            }

            .chat-messages {
                padding: 0.75rem;
            }

            .message-bubble {
                max-width: 90%;
                padding: 0.5rem 0.75rem;
            }

            .message-text {
                font-size: 0.8125rem;
            }

            .message-sender {
                font-size: 0.6875rem;
            }

            .message-time {
                font-size: 0.6875rem;
            }

            .chat-input-section {
                padding: 0.75rem;
            }

            .btn {
                padding: 0.625rem 0.75rem;
                font-size: 0.8125rem;
            }

            .message-input {
                padding: 0.625rem 0.75rem;
                font-size: 0.8125rem;
            }
        }

        /* Safari mobile fix */
        @supports (-webkit-touch-callout: none) {
            .chat-container {
                height: -webkit-fill-available;
            }
        }

        /* Custom scrollbar */
        .chat-messages::-webkit-scrollbar {
            width: 6px;
        }

        .chat-messages::-webkit-scrollbar-track {
            background: #1f2937;
        }

        .chat-messages::-webkit-scrollbar-thumb {
            background: #4b5563;
            border-radius: 3px;
        }

        .chat-messages::-webkit-scrollbar-thumb:hover {
            background: #6b7280;
        }

        /* Animation for new messages */
        @keyframes slideInMessage {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .message-container.new-message {
            animation: slideInMessage 0.3s ease-out;
        }
        .ml-20{
            margin-right: 28px;
        }
    </style>
</head>
<body>
    <div class="chat-container">
        <!-- Chat Header -->
        <div class="chat-header">
            <div class="chat-header-content">
                <h2 class="chat-title">
                    @if(auth('admin')->check())
                        💬 Chat with {{ $chat->user->name }}
                    @else
                        💬 Chat Support
                    @endif
                </h2>
                <div class="chat-status ml-20">
                    @if(!$chat->is_active)
                        <span class="status-badge status-closed">Chat Closed</span>
                    @elseif($chat->admin)
                        <span class="status-badge status-assigned">Assigned: {{ Str::limit($chat->admin->name, 10) }}</span>
                    @else
                        <span class="status-badge status-waiting">Waiting...</span>
                    @endif

                    @if(auth()->check() && !auth('admin')->check())
                        @php
                            $unreadCount = $chat->messages()->where('sender_type', 'admin')->where('is_read', false)->count();
                        @endphp
                        @if($unreadCount > 0)
                            <span class="status-badge status-unread">
                                {{ $unreadCount }} unread
                            </span>
                        @endif
                    @endif
                </div>
            </div>
        </div>

        <!-- Chat Reactivation Notice -->
        @if(auth()->check() && !auth('admin')->check() && $chat->messages()->count() > 0)
            <div class="notice">
                <div class="notice-content">
                    <svg class="notice-icon" viewBox="0 0 20 20" fill="currentColor">
                        <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd" />
                    </svg>
                    <div>
                        <p class="notice-text">
                            Welcome back! You're continuing your previous conversation.
                        </p>
                    </div>
                </div>
            </div>
        @endif

        <!-- Show who closed the chat if closed -->
        @if(!$chat->is_active && $chat->admin)
            <div class="notice bg-red-100 border-l-4 border-red-500 text-red-800 mb-4">
                <div class="notice-content">
                    <svg class="notice-icon" viewBox="0 0 20 20" fill="currentColor">
                        <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd" />
                    </svg>
                    <div>
                        <p class="notice-text">
                            این چت توسط ادمین <b>{{ $chat->admin->name }}</b> بسته شده است.
                        </p>
                    </div>
                </div>
            </div>
        @endif

        <!-- Status Indicator -->
        <div id="status-indicator" class="status-indicator">
            <div class="spinner"></div>
            <span id="status-text" class="status-text">Processing...</span>
        </div>

        <!-- Chat Messages -->
        <div class="chat-messages" id="chat-messages">
            @foreach($messages as $message)
                <div class="message-container {{ $message['sender_type'] === 'admin' ? 'admin' : 'user' }}">
                    <div class="message-bubble message-{{ $message['sender_type'] }}" data-message-id="{{ $message['id'] }}">
                        <div class="message-sender">
                            {{ $message['sender_name'] ?? 'Admin' }}
                        </div>
                        <div class="message-text">{{ $message['message'] }}</div>
                        <div class="message-footer">
                            <div class="message-time">
                                {{ \Illuminate\Support\Carbon::parse($message['created_at'])->format('M d, H:i') }}
                            </div>
                            <div class="message-status status-{{ $message['delivery_status'] ?? 'sent' }}">
                                @if(($message['delivery_status'] ?? 'sent') === 'sent')
                                    ✓
                                @elseif(($message['delivery_status'] ?? 'sent') === 'delivered')
                                    ✓✓
                                @else
                                    ✓✓
                                @endif
                            </div>
                        </div>
                    </div>
                </div>
            @endforeach
        </div>

        <!-- Typing Indicator -->
        <div id="typing-indicator" class="typing-indicator">
            <div class="typing-content">
                <span class="typing-text" id="typing-text">
                    @if(auth('admin')->check())
                        @if(isset($userTyping) && $userTyping)
                            {{ $chat->user->name }} is typing
                        @else
                            Someone is typing
                        @endif
                    @else
                        @if(isset($adminTyping) && $adminTyping)
                            Support is typing
                        @else
                            Someone is typing
                        @endif
                    @endif
                </span>
                <div class="typing-dots">
                    <div class="typing-dot"></div>
                    <div class="typing-dot"></div>
                    <div class="typing-dot"></div>
                </div>
            </div>
        </div>

        <!-- Chat Input -->
        <div class="chat-input-section">
            @if(!$chat->is_active)
                <div class="warning-notice">
                    <div class="warning-content">
                        <svg class="warning-icon" viewBox="0 0 20 20" fill="currentColor">
                            <path fill-rule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clip-rule="evenodd" />
                        </svg>
                        <div>
                            <p class="warning-text">
                                This chat is currently closed. Sending a message will reactivate it.
                            </p>
                        </div>
                    </div>
                </div>
            @endif

            <form id="chat-form" method="POST" action="{{ auth('admin')->check() ? route('admin.chat.send', $chat) : route('chat.send', $chat) }}" class="chat-form">
                @csrf
                <input type="text" name="message" id="message-input" placeholder="Type your message..." class="message-input" required>
                <button type="submit" id="send-button" class="btn btn-primary">Send</button>
                <button type="button" id="refresh-ticks" class="btn btn-secondary">↻</button>
            </form>
        </div>

        <!-- Close Chat Button (Admin Only) -->
        @if(auth('admin')->check() && $chat->is_active)
            <div class="close-chat-section">
                <form method="POST" action="{{ route('admin.chat.close', $chat) }}">
                    @csrf
                    <button type="submit" id="close-button" class="btn btn-danger">
                        Close Chat
                    </button>
                </form>
            </div>
        @endif
    </div>

    @include('admin.layouts.scripts')
    <script>
        // مقدار chatId را از blade بگیر
        const chatId = {{ $chat->id }};
        // مقدار CSRF را از meta بگیر
        const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
        // مقدار نام کاربر جاری
        const currentUser = @json(auth('admin')->check() ? ($chat->admin->name ?? 'Support') : $chat->user->name);
        let lastMessageId = {{ collect($messages)->max('id') ?? 0 }};
        const authIsAdmin = {{ auth('admin')->check() ? 'true' : 'false' }};
        let chatTabActive = true;
        let isChatPage = window.location.pathname.match(/chat\/(\d+)/) !== null;

        document.addEventListener('visibilitychange', function() {
            chatTabActive = document.visibilityState === 'visible';
        });

        let baseChatUrl = window.location.pathname.includes('/admin/') ? `/admin/chat/${chatId}` : `/chat/${chatId}`;

        // تابع اضافه کردن پیام به DOM
        function addMessageToChat(message, scroll = false) {
            const chatBox = document.getElementById('chat-messages');
            // اگر پیام با این id قبلاً وجود دارد، دیگر اضافه نکن
            if (chatBox.querySelector(`[data-message-id="${message.id}"]`)) {
                return;
            }

            const msgDiv = document.createElement('div');
            msgDiv.className = 'message-container new-message ' + (message.sender_type === 'admin' ? 'admin' : 'user');

            // تعیین رنگ و آیکون تیک
            let statusClass = 'status-sent', statusIcon = '✓';
            if (message.delivery_status === 'delivered') {
                statusClass = 'status-delivered'; statusIcon = '✓✓';
            } else if (message.delivery_status === 'read') {
                statusClass = 'status-read'; statusIcon = '✓✓';
            }

            // فقط برای پیام‌های خود کاربر تیک نمایش بده
            const isCurrentUser = (authIsAdmin && message.sender_type === 'admin') || (!authIsAdmin && message.sender_type === 'user');
            let readStatusHtml = '';
            if (isCurrentUser) {
                readStatusHtml = `<div class="message-status ${statusClass}">${statusIcon}</div>`;
            }

            msgDiv.innerHTML = `
                <div class="message-bubble message-${message.sender_type}" data-message-id="${message.id}">
                    <div class="message-sender">${message.sender_name || (message.sender_type === 'admin' ? 'Support' : currentUser)}</div>
                    <div class="message-text">${message.message}</div>
                    <div class="message-footer">
                        <div class="message-time">${new Date(message.created_at).toLocaleString('fa-IR')}</div>
                        ${readStatusHtml}
                    </div>
                </div>
            `;

            chatBox.appendChild(msgDiv);
            if (scroll) chatBox.scrollTop = chatBox.scrollHeight;
        }

        // تابع بررسی اسکرول پایین بودن چت
        function isScrolledToBottom() {
            const chatBox = document.getElementById('chat-messages');
            if (!chatBox) return false;
            return chatBox.scrollHeight - chatBox.scrollTop - chatBox.clientHeight < 10;
        }

        // تابع دریافت پیام‌های جدید بدون پاک کردن قبلی‌ها
        function fetchNewMessages() {
            fetch(`${baseChatUrl}/new-messages?include_all=true`)
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        const chatBox = document.getElementById('chat-messages');
                        let shouldScroll = isScrolledToBottom();
                        let maxId = lastMessageId;
                        data.messages.forEach(msg => {
                            // اگر پیام وجود ندارد، اضافه کن
                            let msgDiv = chatBox.querySelector(`[data-message-id="${msg.id}"]`);
                            if (!msgDiv) {
                                addMessageToChat(msg, false);
                                if (msg.id > maxId) maxId = msg.id;
                            } else {
                                // اگر پیام وجود دارد، فقط وضعیت تیک را آپدیت کن
                                let statusDiv = msgDiv.querySelector('.message-status');
                                let statusClass = 'status-sent', statusIcon = '✓';
                                if (msg.delivery_status === 'delivered') {
                                    statusClass = 'status-delivered'; statusIcon = '✓✓';
                                } else if (msg.delivery_status === 'read') {
                                    statusClass = 'status-read'; statusIcon = '✓✓';
                                }
                                if (statusDiv) {
                                    statusDiv.className = 'message-status ' + statusClass;
                                    statusDiv.innerHTML = statusIcon;
                                }
                            }
                        });
                        lastMessageId = maxId;
                        if (data.messages.length > 0 && shouldScroll) {
                            chatBox.scrollTop = chatBox.scrollHeight;
                        }
                        // فقط اگر پیام جدید از طرف مقابل آمده و هنوز خوانده نشده، mark-read بزن (بدون توجه به اسکرول)
                        const urlChatId = getChatIdFromUrl();
                        const unreadFromOther = data.messages.some(msg =>
                            ((authIsAdmin && msg.sender_type === 'user') || (!authIsAdmin && msg.sender_type === 'admin')) &&
                            msg.delivery_status !== 'read'
                        );
                        if (
                            urlChatId === chatId &&
                            chatTabActive &&
                            document.getElementById('chat-messages') !== null &&
                            unreadFromOther &&
                            (
                                (!authIsAdmin && window.location.pathname === `/chat/${chatId}`) ||
                                (authIsAdmin && window.location.pathname === `/admin/chat/${chatId}/view`)
                            )
                        ) {
                            fetch(`${baseChatUrl}/mark-read`, {
                                method: 'POST',
                                headers: {
                                    'X-CSRF-TOKEN': csrfToken
                                }
                            });
                        }
                    }
                })
                .catch(err => console.error('Polling error:', err));
        }

        let pollingInterval = null;

        function startPolling() {
            if (pollingInterval) return;
            pollingInterval = setInterval(fetchNewMessages, 2000);
        }

        function stopPolling() {
            if (pollingInterval) {
                clearInterval(pollingInterval);
                pollingInterval = null;
            }
        }

        // فقط اگر واقعا در چت هستیم polling را فعال کن
        function checkAndStartPolling() {
            if (isChatReallyActive()) {
                startPolling();
            } else {
                stopPolling();
            }
        }

        document.addEventListener('visibilitychange', checkAndStartPolling);
        window.addEventListener('popstate', checkAndStartPolling);
        window.addEventListener('focus', checkAndStartPolling);
        window.addEventListener('blur', checkAndStartPolling);
        window.addEventListener('beforeunload', function() {
            stopPolling();
            lastMessageId = 0;
        });

        // در ابتدا هم بررسی کن
        checkAndStartPolling();

        // ارسال پیام با AJAX
        document.getElementById('chat-form').addEventListener('submit', function(e) {
            e.preventDefault();
            const messageInput = document.getElementById('message-input');
            const message = messageInput.value;
            if (!message.trim()) return;
            messageInput.value = '';
            // دکمه Send را غیرفعال نکن و هیچ مکثی ایجاد نکن
            fetch(this.action, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                    'X-CSRF-TOKEN': csrfToken
                },
                body: new URLSearchParams({message})
            }).then(() => {
                fetchNewMessages();
            }).catch(err => {
                console.error('Send message error:', err);
            });
        });

        function getChatIdFromUrl() {
            const path = window.location.pathname;
            const userChatRegex = /^\/chat\/(\d+)$/;
            const adminChatRegex = /^\/admin\/chat\/(\d+)\/view$/;
            let match = path.match(userChatRegex) || path.match(adminChatRegex);
            if (!match) return null;
            return parseInt(match[1]);
        }

        function isChatReallyActive() {
            // فقط اگر دقیقا در صفحه چت کاربر یا چت مدیر هستیم و عنصر چت وجود دارد و تب فعال است و chatId درست است
            const urlChatId = getChatIdFromUrl();
            if (!urlChatId) return false;
            if (urlChatId !== chatId) return false;
            return chatTabActive && document.getElementById('chat-messages') !== null;
        }

        // در ابتدا پیام‌های اولیه را به DOM اضافه کن و اسکرول کن پایین
        window.addEventListener('DOMContentLoaded', function() {
            const chatBox = document.getElementById('chat-messages');
            @foreach($messages as $message)
                addMessageToChat(@json($message), false);
                lastMessageId = {{ $message['id'] }};
            @endforeach
            chatBox.scrollTop = chatBox.scrollHeight;
        });
    </script>
</body>
</html>
