@section('title', ($product->seo_title ?? $product->name))
@section('meta_description', ($product->meta_description ?? Str::limit($product->description, 150)) . ' Buy this product at ' . config('app.name') . ' with fast delivery and secure checkout.')
@section('meta_keywords', $product->seo_keywords)

<!-- Open Graph Meta Tags -->
@section('og_title', $product->og_title)
@section('og_description', $product->og_description)
@section('og_image', $product->og_image ? asset($product->og_image) : asset($product->first_image))
@section('og_type', 'product')
@section('og_url', $product->canonical_url)

<!-- Twitter Card Meta Tags -->
@section('twitter_card', 'summary_large_image')
@section('twitter_title', $product->twitter_title)
@section('twitter_description', $product->twitter_description)
@section('twitter_image', $product->twitter_image ? asset($product->twitter_image) : asset($product->first_image))

<!-- Additional SEO Meta Tags -->
@section('canonical_url', $product->canonical_url)
@section('robots', $product->seo_indexable ? 'index, follow' : 'noindex, nofollow')

<!-- Schema.org Structured Data -->
@section('schema_markup')
<script type="application/ld+json">
{
    "@context": "https://schema.org",
    "@type": "Product",
    "name": "{{ $product->name }}",
    "description": "{{ strip_tags($product->description) }}",
    "image": [
        @foreach($product->images as $image)
            "{{ asset($image) }}"@if(!$loop->last),@endif
        @endforeach
    ],
    "brand": {
        "@type": "Brand",
        "name": "{{ $product->brand ?: config('app.name') }}"
    },
    @if($product->sku)
    "sku": "{{ $product->sku }}",
    @endif
    @if($product->gtin)
    "gtin": "{{ $product->gtin }}",
    @endif
    @if($product->mpn)
    "mpn": "{{ $product->mpn }}",
    @endif
    "category": "{{ $product->category->name ?? '' }}",
    "condition": "{{ $product->condition }}",
    "availability": "{{ $product->availability === 'in_stock' ? 'https://schema.org/InStock' : 'https://schema.org/OutOfStock' }}",
    "price": "{{ $product->discounted_price }}",
    "priceCurrency": "USD",
    "url": "{{ $product->canonical_url }}",
    @if($product->rating_value && $product->rating_count > 0)
    "aggregateRating": {
        "@type": "AggregateRating",
        "ratingValue": "{{ $product->rating_value }}",
        "reviewCount": "{{ $product->rating_count }}",
        "bestRating": "5",
        "worstRating": "1"
    },
    @endif
    "offers": {
        "@type": "Offer",
        "price": "{{ $product->discounted_price }}",
        "priceCurrency": "USD",
        "availability": "{{ $product->availability === 'in_stock' ? 'https://schema.org/InStock' : 'https://schema.org/OutOfStock' }}",
        "url": "{{ $product->canonical_url }}",
        "priceValidUntil": "{{ now()->addYear()->format('Y-m-d') }}"
    }
}
</script>
@endsection

<x-app-layout>
    <x-slot name="header">
        <div class="flex flex-col sm:flex-row justify-between items-start sm:items-center space-y-2 sm:space-y-0">
            <h2 class="font-bold text-lg sm:text-xl md:text-2xl text-gray-900 leading-tight">
                <span class="bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
                    {{ $product->name }}
                </span>
            </h2>
            <a href="{{ route('products.index') }}" class="inline-flex items-center px-4 py-2 bg-gradient-to-r from-gray-600 to-gray-700 hover:from-gray-700 hover:to-gray-800 text-white font-semibold rounded-lg transition-all duration-300 transform hover:scale-105 shadow-md hover:shadow-lg text-sm">
                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18"/>
                </svg>
                {{ __('Back to Products') }}
            </a>
        </div>
    </x-slot>

    <!-- Success/Error Messages - Compact -->
    @if(session('success'))
        <div class="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 pt-4">
            <div class="bg-gradient-to-r from-green-400 to-green-600 text-white px-4 py-3 rounded-lg shadow-md animate-fade-in">
                <div class="flex items-center">
                    <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                    <span class="font-semibold text-sm">{{ session('success') }}</span>
                </div>
            </div>
        </div>
    @endif

    @if(session('error'))
        <div class="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 pt-4">
            <div class="bg-gradient-to-r from-red-400 to-red-600 text-white px-4 py-3 rounded-lg shadow-md animate-fade-in">
                <div class="flex items-center">
                    <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                    <span class="font-semibold text-sm">{{ session('error') }}</span>
                </div>
            </div>
        </div>
    @endif

    <div class="py-8 bg-gradient-to-br from-gray-50 to-blue-50">
        <div class="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
            <!-- Breadcrumb Navigation - Compact -->
            <nav class="mb-6" aria-label="Breadcrumb">
                <ol class="flex items-center space-x-1 text-xs sm:text-sm text-gray-600">
                    <li>
                        <a href="{{ route('home') }}" class="hover:text-blue-600 transition-colors">
                            {{ __('Home') }}
                        </a>
                    </li>
                    <li class="flex items-center">
                        <svg class="w-3 h-3 sm:w-4 sm:h-4 mx-1 sm:mx-2" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z" clip-rule="evenodd"/>
                        </svg>
                        <a href="{{ route('products.index') }}" class="hover:text-blue-600 transition-colors">
                            {{ __('Products') }}
                        </a>
                    </li>
                    @if($product->category)
                    <li class="flex items-center">
                        <svg class="w-3 h-3 sm:w-4 sm:h-4 mx-1 sm:mx-2" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z" clip-rule="evenodd"/>
                        </svg>
                        <a href="{{ route('products.index', ['category' => $product->category->id]) }}" class="hover:text-blue-600 transition-colors">
                            {{ $product->category->name }}
                        </a>
                    </li>
                    @endif
                    <li class="flex items-center">
                        <svg class="w-3 h-3 sm:w-4 sm:h-4 mx-1 sm:mx-2" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z" clip-rule="evenodd"/>
                        </svg>
                        <span class="text-gray-900 font-medium">{{ $product->name }}</span>
                    </li>
                </ol>
            </nav>

            <!-- Main Product Section - Compact -->
            <div class="bg-white rounded-2xl shadow-lg overflow-hidden border border-gray-100">
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-0">
                    <!-- Product Image - Compact -->
                    <div class="relative group">
                        @if($product->image)
                            <div class="grid grid-cols-1 gap-3">
                                <!-- Main Image -->
                                <div class="relative">
                                    @php
                                        $imageAltTexts = $product->image_alt_texts ?? [];
                                        $mainImageAlt = !empty($imageAltTexts) ? $imageAltTexts[0] : $product->name . ' - ' . config('app.name');
                                    @endphp
                                    <img src="{{ asset($product->first_image) }}" alt="{{ $mainImageAlt }}"
                                         id="main-product-image"
                                         class="w-full h-64 sm:h-80 lg:h-96 object-cover rounded-lg transition-transform duration-300 group-hover:scale-105">
                                </div>

                                <!-- Thumbnail Gallery - Compact -->
                                @if(count(json_decode($product->image, true)) > 1)
                                    <div class="grid grid-cols-4 gap-2 sm:gap-3 mt-3">
                                        @foreach(json_decode($product->image, true) as $index => $image)
                                            @php
                                                $thumbnailAlt = isset($imageAltTexts[$index]) ? $imageAltTexts[$index] : $product->name . ' - Image ' . ($index + 1) . ' - ' . config('app.name');
                                            @endphp
                                            <div class="relative cursor-pointer group/thumb">
                                                <img src="{{ asset($image) }}"
                                                     alt="{{ $thumbnailAlt }}"
                                                     class="thumbnail-image w-full h-16 sm:h-20 object-cover rounded-md transition-all duration-300 group-hover/thumb:scale-110"
                                                     onclick="document.getElementById('main-product-image').src = '{{ asset($image) }}'; document.getElementById('main-product-image').alt = '{{ $thumbnailAlt }}'">
                                            </div>
                                        @endforeach
                                    </div>
                                @endif
                            </div>
                        @else
                            <div class="w-full h-64 sm:h-80 lg:h-96 bg-gradient-to-br from-gray-100 to-gray-200 flex items-center justify-center">
                                <div class="text-center">
                                    <svg class="w-16 h-16 sm:w-20 sm:h-20 text-gray-300 mx-auto mb-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"/>
                                    </svg>
                                    <span class="text-gray-400 text-sm sm:text-base font-medium">{{ __('No image available') }}</span>
                                </div>
                            </div>
                        @endif

                        <!-- Floating Stock Badge - Compact -->
                        <div class="absolute top-3 left-3 sm:top-4 sm:left-4">
                            @if($product->stock > 10)
                                <span class="inline-flex items-center px-2 py-1 sm:px-3 sm:py-1.5 rounded-full text-xs font-bold bg-gradient-to-r from-green-400 to-green-600 text-white shadow-md">
                                    <svg class="w-3 h-3 sm:w-4 sm:h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                        <path fill-rule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clip-rule="evenodd"/>
                                    </svg>
                                    {{ __('In Stock') }}
                                </span>
                            @elseif($product->stock > 0)
                                <span class="inline-flex items-center px-2 py-1 sm:px-3 sm:py-1.5 rounded-full text-xs font-bold bg-gradient-to-r from-yellow-400 to-orange-500 text-white shadow-md animate-pulse">
                                    <svg class="w-3 h-3 sm:w-4 sm:h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                        <path fill-rule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clip-rule="evenodd"/>
                                    </svg>
                                    {{ __('Low Stock') }}
                                </span>
                            @else
                                <span class="inline-flex items-center px-2 py-1 sm:px-3 sm:py-1.5 rounded-full text-xs font-bold bg-gradient-to-r from-red-400 to-red-600 text-white shadow-md">
                                    <svg class="w-3 h-3 sm:w-4 sm:h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                        <path fill-rule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z" clip-rule="evenodd"/>
                                    </svg>
                                    {{ __('Out of Stock') }}
                                </span>
                            @endif
                        </div>
                    </div>

                    <!-- Product Details - Compact -->
                    <div class="p-4 sm:p-6 lg:p-8">
                        <!-- Product Header -->
                        <div class="mb-4 sm:mb-6">
                            <h1 class="text-2xl sm:text-3xl lg:text-4xl font-black text-gray-900 mb-3 leading-tight">
                                {{ $product->name }}
                            </h1>

                            <!-- Product Meta Information - Compact -->
                            <div class="flex flex-wrap items-center gap-2 sm:gap-3 text-xs sm:text-sm text-gray-600 mb-3">
                                @if($product->brand)
                                <div class="flex items-center">
                                    <span class="font-medium">{{ __('Brand:') }}</span>
                                    <span class="ml-1">{{ $product->brand }}</span>
                                </div>
                                @endif
                                @if($product->sku)
                                <div class="flex items-center">
                                    <span class="font-medium">{{ __('SKU:') }}</span>
                                    <span class="ml-1">{{ $product->sku }}</span>
                                </div>
                                @endif
                                @if($product->condition)
                                <div class="flex items-center">
                                    <span class="font-medium">{{ __('Condition:') }}</span>
                                    <span class="ml-1 capitalize">{{ $product->condition }}</span>
                                </div>
                                @endif
                                @if($product->category)
                                <div class="flex items-center">
                                    <span class="font-medium">{{ __('Category:') }}</span>
                                    <a href="{{ route('products.index', ['category' => $product->category->id]) }}" class="ml-1 text-blue-600 hover:underline">
                                        {{ $product->category->name }}
                                    </a>
                                </div>
                                @endif
                            </div>

                            <!-- Product Description - Compact -->
                            <div class="mb-4 sm:mb-6">
                                <h2 class="text-base sm:text-lg font-semibold text-gray-900 mb-2">{{ __('Description') }}</h2>
                                <p class="text-sm sm:text-base text-gray-700 leading-relaxed">{{ $product->description }}</p>
                            </div>

                            @if($product->dynamic_fields)
                            <div class="mb-4">
                                <h3 class="text-base sm:text-lg font-medium text-gray-900 mb-3">Product Specifications</h3>
                                <div class="mt-2 grid grid-cols-1 gap-3 sm:grid-cols-2">
                                    @php
                                        $dynamicFields = [];
                                        if (is_string($product->dynamic_fields)) {
                                            $decoded = json_decode($product->dynamic_fields, true);
                                            if (is_array($decoded)) {
                                                $dynamicFields = $decoded;
                                            }
                                        } elseif (is_array($product->dynamic_fields)) {
                                            $dynamicFields = $product->dynamic_fields;
                                        }
                                    @endphp

                                    @foreach($dynamicFields as $index => $field)
                                        <div class="bg-white overflow-hidden shadow rounded-lg">
                                            <div class="px-3 py-4 sm:px-4 sm:py-5">
                                                <dt class="text-xs sm:text-sm font-medium text-gray-500 truncate mb-2 sm:mb-3">{{ $field['title'] ?? 'Unknown Field' }}</dt>
                                                <dd class="mt-1 text-xs sm:text-sm text-gray-900">
                                                    @if($field['type'] === 'text')
                                                        <input type="text"
                                                               name="dynamic_field_{{ $index }}"
                                                               class="w-full px-2 py-1.5 sm:px-3 sm:py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent text-sm"
                                                               placeholder="Enter {{ $field['title'] }}">
                                                    @elseif($field['type'] === 'select')
                                                        @if(isset($field['options']) && is_array($field['options']))
                                                            <select name="dynamic_field_{{ $index }}"
                                                                    class="dynamic-field-select w-full px-2 py-1.5 sm:px-3 sm:py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent text-sm"
                                                                    data-field-index="{{ $index }}">
                                                                        <option value="">Select {{ $field['title'] }}</option>
                                                                        @foreach($field['options'] as $optionIndex => $option)
                                                                            <option value="{{ $optionIndex }}"
                                                                                    data-price="{{ $option['price'] ?? 0 }}">
                                                                                {{ $option['name'] ?? $option }}
                                                                                @if(isset($option['price']) && $option['price'] > 0)
                                                                                    (+${{ number_format($option['price'], 2) }})
                                                                                @endif
                                                                            </option>
                                                                        @endforeach
                                                                    </select>
                                                        @else
                                                            <span class="text-gray-500 text-xs sm:text-sm">No options available</span>
                                                        @endif
                                                    @elseif($field['type'] === 'checkbox')
                                                        @if(isset($field['options']) && is_array($field['options']))
                                                            <div class="space-y-1 sm:space-y-2">
                                                                @foreach($field['options'] as $optionIndex => $option)
                                                                    <label class="flex items-center">
                                                                        <input type="checkbox"
                                                                               name="dynamic_field_{{ $index }}[]"
                                                                               value="{{ $optionIndex }}"
                                                                               class="dynamic-field-checkbox mr-2 h-3 w-3 sm:h-4 sm:w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                                                                               data-price="{{ $option['price'] ?? 0 }}">
                                                                        <span class="text-xs sm:text-sm text-gray-700">{{ $option['name'] ?? $option }}</span>
                                                                        @if(isset($option['price']) && $option['price'] > 0)
                                                                            <span class="text-green-600 font-semibold ml-1 sm:ml-2 text-xs sm:text-sm">+${{ number_format($option['price'], 2) }}</span>
                                                                        @endif
                                                                    </label>
                                                                @endforeach
                                                            </div>
                                                        @else
                                                            <span class="text-gray-500 text-xs sm:text-sm">No options available</span>
                                                        @endif
                                                    @elseif($field['type'] === 'radio')
                                                        @if(isset($field['options']) && is_array($field['options']))
                                                            <div class="space-y-1 sm:space-y-2">
                                                                @foreach($field['options'] as $optionIndex => $option)
                                                                    <label class="flex items-center">
                                                                        <input type="radio"
                                                                               name="dynamic_field_{{ $index }}"
                                                                               value="{{ $optionIndex }}"
                                                                               class="dynamic-field-radio mr-2 h-3 w-3 sm:h-4 sm:w-4 text-blue-600 focus:ring-blue-500 border-gray-300"
                                                                               data-price="{{ $option['price'] ?? 0 }}">
                                                                        <span class="text-xs sm:text-sm text-gray-700">{{ $option['name'] ?? $option }}</span>
                                                                        @if(isset($option['price']) && $option['price'] > 0)
                                                                            <span class="text-green-600 font-semibold ml-1 sm:ml-2 text-xs sm:text-sm">+${{ number_format($option['price'], 2) }}</span>
                                                                        @endif
                                                                    </label>
                                                                @endforeach
                                                            </div>
                                                        @else
                                                            <span class="text-gray-500 text-xs sm:text-sm">No options available</span>
                                                        @endif
                                                    @endif
                                                </dd>
                                            </div>
                                        </div>
                                    @endforeach
                                </div>
                            </div>
                            @endif

                            <div class="mb-4">
                                <h3 class="text-base sm:text-lg font-medium text-gray-900 mb-3">Product Images</h3>
                                <div class="mt-2 grid grid-cols-2 gap-2 sm:gap-3 sm:grid-cols-3 md:grid-cols-4">
                                    @php
                                        $productImages = [];
                                        if ($product->image) {
                                            if (is_string($product->image)) {
                                                $decoded = json_decode($product->image, true);
                                                if (is_array($decoded)) {
                                                    $productImages = $decoded;
                                                } else {
                                                    $productImages = [$product->image];
                                                }
                                            } elseif (is_array($product->image)) {
                                                $productImages = $product->image;
                                            }
                                        }
                                    @endphp

                                    @foreach($productImages as $image)
                                        <div class="relative">
                                            <img src="{{ asset('/' . $image) }}" alt="{{ $product->name }}" class="w-full h-20 sm:h-24 object-cover rounded-lg">
                                        </div>
                                    @endforeach
                                </div>
                            </div>
                        </div>

                        <div class="mb-6 sm:mb-8">
                            <div class="flex flex-wrap items-baseline space-x-2 sm:space-x-4">
                                @if($product->hasDiscount())
                                    <span class="text-2xl sm:text-3xl lg:text-4xl font-black bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent price-display">
                                        ${{ number_format($product->discounted_price, 2) }}
                                    </span>
                                    <span class="text-lg sm:text-xl text-gray-400 line-through">
                                        ${{ number_format($product->price, 2) }}
                                    </span>
                                    <span class="text-sm sm:text-lg text-red-500 font-semibold">
                                        {{ $product->discount_percentage }}% OFF
                                    </span>
                                @else
                                <span class="text-2xl sm:text-3xl lg:text-4xl font-black bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent price-display">
                                    ${{ number_format($product->price, 2) }}
                                </span>
                                @endif
                                @if($product->stock <= 5 && $product->stock > 0)
                                    <span class="text-sm sm:text-lg text-orange-500 font-semibold">
                                        {{ __('Only') }} {{ $product->stock }} {{ __('left!') }}
                                    </span>
                                @endif
                            </div>
                            <div id="additional-costs" class="mt-2 text-xs sm:text-sm text-green-600 hidden">
                                <span id="additional-costs-text"></span>
                            </div>
                        </div>

                        <!-- Add to Cart Section - Compact -->
                        @auth
                            @if($product->stock > 0)
                                <form action="{{ route('cart.add', $product) }}" method="POST" class="mb-6 sm:mb-8">
                                    @csrf
                                    <input type="hidden" name="total_price" id="total_price" value="{{ $product->hasDiscount() ? $product->discounted_price : $product->price }}">
                                    <input type="hidden" name="dynamic_selections" id="dynamic_selections" value="">

                                    <div class="space-y-3 sm:space-y-4">
                                        <div class="flex flex-col sm:flex-row sm:items-center space-y-2 sm:space-y-0 sm:space-x-4">
                                            <label for="quantity" class="text-sm sm:text-lg font-semibold text-gray-700">
                                                {{ __('Quantity:') }}
                                            </label>
                                            <input type="number"
                                                   name="quantity"
                                                   id="quantity"
                                                   value="1"
                                                   min="1"
                                                   max="{{ $product->stock }}"
                                                   class="w-full sm:w-24 px-3 py-2 sm:px-4 sm:py-3 rounded-lg sm:rounded-xl border-2 border-gray-200 shadow-sm focus:border-blue-500 focus:ring-4 focus:ring-blue-200 transition-all duration-300 text-base sm:text-lg font-semibold text-center">
                                        </div>
                                        <button type="submit" class="w-full inline-flex items-center justify-center px-6 py-3 sm:px-8 sm:py-4 text-base sm:text-lg font-semibold text-white bg-gradient-to-r from-blue-600 to-purple-600 rounded-lg sm:rounded-xl hover:from-blue-700 hover:to-purple-700 focus:outline-none focus:ring-4 focus:ring-blue-300 transform hover:scale-105 transition-all duration-300 shadow-md hover:shadow-lg">
                                            <svg class="w-5 h-5 sm:w-6 sm:h-6 mr-2 sm:mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4m0 0L7 13m0 0l-2.5 5M7 13l2.5 5m0 0h7M9.5 18a1.5 1.5 0 003 0m6 0a1.5 1.5 0 003 0"/>
                                            </svg>
                                            {{ __('Add to Cart') }}
                                        </button>
                                    </div>
                                </form>
                            @endif
                        @else
                            <div class="mb-6 sm:mb-8 p-4 sm:p-6 bg-gradient-to-r from-blue-50 to-purple-50 rounded-lg sm:rounded-xl border border-blue-200">
                                <div class="text-center">
                                    <svg class="w-8 h-8 sm:w-12 sm:h-12 text-blue-500 mx-auto mb-2 sm:mb-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
                                    </svg>
                                    <p class="text-sm sm:text-lg text-gray-700 mb-3 sm:mb-4">{{ __('Please login to add items to your cart') }}</p>
                                    <a href="{{ route('login') }}" class="inline-flex items-center px-4 py-2 sm:px-6 sm:py-3 bg-gradient-to-r from-blue-600 to-purple-600 text-white font-semibold rounded-lg sm:rounded-xl hover:from-blue-700 hover:to-purple-700 transition-all duration-300 transform hover:scale-105 text-sm sm:text-base">
                                        {{ __('Login Now') }}
                                    </a>
                                </div>
                            </div>
                        @endauth

                        <!-- Product Specifications - Compact -->
                        <div class="grid grid-cols-1 sm:grid-cols-2 gap-4 sm:gap-6 p-4 sm:p-6 bg-gray-50 rounded-lg sm:rounded-xl">
                            <div class="space-y-2 sm:space-y-3">
                                <div class="flex items-center">
                                    <svg class="w-4 h-4 sm:w-5 sm:h-5 text-blue-500 mr-2 sm:mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 7h.01M7 3h5c.512 0 1.024.195 1.414.586l7 7a2 2 0 010 2.828l-7 7a2 2 0 01-2.828 0l-7-7A1.994 1.994 0 013 12V7a4 4 0 014-4z"/>
                                    </svg>
                                    <span class="text-xs sm:text-sm text-gray-600">
                                        <span class="font-semibold">{{ __('Category:') }}</span> {{ $product->category?->name ?? 'Uncategorized' }}
                                    </span>
                                </div>
                                <div class="flex items-center">
                                    <svg class="w-4 h-4 sm:w-5 sm:h-5 text-green-500 mr-2 sm:mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4"/>
                                    </svg>
                                    <span class="text-xs sm:text-sm text-gray-600">
                                        <span class="font-semibold">{{ __('Stock:') }}</span> {{ $product->stock }} {{ __('items') }}
                                    </span>
                                </div>
                                <div class="flex items-center">
                                    <svg class="w-4 h-4 sm:w-5 sm:h-5 text-yellow-500 mr-2 sm:mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                                    </svg>
                                    <span class="text-xs sm:text-sm text-gray-600">
                                        <span class="font-semibold">{{ __('Price:') }}</span> ${{ number_format($product->price, 2) }}
                                    </span>
                                </div>
                            </div>
                            <div class="space-y-2 sm:space-y-3">
                                <div class="flex items-center">
                                    <svg class="w-4 h-4 sm:w-5 sm:h-5 text-purple-500 mr-2 sm:mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
                                    </svg>
                                    <span class="text-xs sm:text-sm text-gray-600">
                                        <span class="font-semibold">{{ __('Seller:') }}</span> {{ $product->user?->name ?? 'Unknown' }}
                                    </span>
                                </div>
                                <div class="flex items-center">
                                    <svg class="w-4 h-4 sm:w-5 sm:h-5 text-orange-500 mr-2 sm:mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7V3a4 4 0 118 0v4m-4 6l-3 5h6l-3-5z"/>
                                    </svg>
                                    <span class="text-xs sm:text-sm text-gray-600">
                                        <span class="font-semibold">{{ __('Added:') }}</span>
                                        {{ $product->created_at ? $product->created_at->format('M d, Y') : 'N/A' }}
                                    </span>
                                </div>
                                <div class="flex items-center">
                                    <svg class="w-4 h-4 sm:w-5 sm:h-5 text-red-500 mr-2 sm:mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                                    </svg>
                                    <span class="text-xs sm:text-sm text-gray-600">
                                        <span class="font-semibold">{{ __('Status:') }}</span>
                                        @if($product->is_approved)
                                            <span class="text-green-600">{{ __('Approved') }}</span>
                                        @else
                                            <span class="text-red-600">{{ __('Pending Approval') }}</span>
                                        @endif
                                    </span>
                                </div>
                            </div>
                        </div>

                        <div class="mt-4 sm:mt-6 p-4 sm:p-6 bg-white rounded-lg sm:rounded-xl shadow-sm">
                            <h3 class="text-base sm:text-lg font-bold text-gray-900 mb-3 sm:mb-4">{{ __('Product Description') }}</h3>
                            <p class="text-xs sm:text-sm text-gray-600 leading-relaxed">{{ $product->description }}</p>
                        </div>

                        @if($product->is_featured)
                            <div class="mt-4 sm:mt-6 p-3 sm:p-4 bg-gradient-to-r from-yellow-50 to-orange-50 rounded-lg sm:rounded-xl border border-yellow-200">
                                <div class="flex items-center text-yellow-700">
                                    <svg class="w-4 h-4 sm:w-5 sm:h-5 mr-2" fill="currentColor" viewBox="0 0 20 20">
                                        <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z"/>
                                    </svg>
                                    <span class="font-semibold text-sm sm:text-base">{{ __('Featured Product') }}</span>
                                </div>
                            </div>
                        @endif
                    </div>
                </div>
            </div>

            <!-- Reviews Section -->
            <div class="mt-12 bg-white rounded-3xl shadow-2xl overflow-hidden border border-gray-100">
                <div class="p-8 lg:p-12">
                    <div class="text-center mb-12">
                        <h2 class="text-3xl lg:text-4xl font-black text-gray-900 mb-4">
                            <span class="bg-gradient-to-r from-yellow-400 to-orange-500 bg-clip-text text-transparent">
                                {{ __('Customer Reviews') }}
                            </span>
                        </h2>
                        <div class="w-24 h-1 bg-gradient-to-r from-yellow-400 to-orange-500 mx-auto rounded-full"></div>
                    </div>

                    @auth
                        @if(!$product->reviews->where('user_id', auth()->id())->count())
                            <div class="mt-12">
                                <h3 class="text-2xl font-bold text-gray-900 mb-6">{{ __('Write a Review') }}</h3>
                                <form action="{{ route('reviews.store', $product) }}" method="POST" class="space-y-6">
                                    @csrf

                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">{{ __('Rating') }}</label>
                                        <div class="flex items-center space-x-2">
                                            @for($i = 1; $i <= 5; $i++)
                                                <input type="radio" name="rating" value="{{ $i }}" id="rating_{{ $i }}" class="sr-only">
                                                <label for="rating_{{ $i }}" class="cursor-pointer">
                                                    <svg class="w-8 h-8 text-gray-300 hover:text-yellow-400 transition-colors" fill="currentColor" viewBox="0 0 20 20">
                                                        <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z"/>
                                                    </svg>
                                                </label>
                                            @endfor
                                        </div>
                                    </div>

                                    <div>
                                        <label for="comment" class="block text-sm font-medium text-gray-700 mb-2">{{ __('Comment') }}</label>
                                        <textarea name="comment" id="comment" rows="4" class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent" placeholder="{{ __('Share your experience with this product...') }}"></textarea>
                                    </div>

                                    <button type="submit" class="inline-flex items-center px-6 py-3 bg-gradient-to-r from-blue-600 to-purple-600 text-white font-semibold rounded-xl hover:from-blue-700 hover:to-purple-700 transition-all duration-300 transform hover:scale-105">
                                        {{ __('Submit Review') }}
                                    </button>
                                </form>
                            </div>
                        @endif
                    @endauth

                    @if($product->reviews->where('is_approved', true)->count() > 0)
                        <div class="grid gap-8 md:grid-cols-2 lg:grid-cols-3">
                            @foreach($product->reviews->where('is_approved', true) as $review)
                                <div class="bg-gradient-to-br from-gray-50 to-gray-100 rounded-2xl p-6 shadow-lg">
                                    <div class="flex items-center mb-4">
                                        <div class="flex-shrink-0">
                                            <div class="w-12 h-12 bg-gradient-to-r from-blue-500 to-purple-600 rounded-full flex items-center justify-center">
                                                <span class="text-white font-bold text-lg">{{ substr($review->user->name, 0, 1) }}</span>
                                            </div>
                                        </div>
                                        <div class="ml-4">
                                            <h4 class="text-lg font-semibold text-gray-900">{{ $review->user->name }}</h4>
                                            <div class="flex items-center">
                                                @for($i = 1; $i <= 5; $i++)
                                                    <svg class="w-4 h-4 {{ $i <= $review->rating ? 'text-yellow-400' : 'text-gray-300' }}" fill="currentColor" viewBox="0 0 20 20">
                                                        <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z"/>
                                                    </svg>
                                                @endfor
                                            </div>
                                        </div>
                                    </div>
                                    <p class="text-gray-600 leading-relaxed">{{ $review->comment }}</p>
                                    <div class="mt-4 text-sm text-gray-500">
                                        {{ $review->created_at->format('M d, Y') }}
                                    </div>
                                </div>
                            @endforeach
                        </div>
                    @else
                        <div class="text-center py-12">
                            <svg class="w-16 h-16 text-gray-300 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"/>
                            </svg>
                            <p class="text-gray-500 text-lg">{{ __('No reviews yet. Be the first to review this product!') }}</p>
                        </div>
                    @endif
                </div>
            </div>

            <!-- Related Products -->
            @if($relatedProducts->isNotEmpty())
                <div class="mt-12 bg-white rounded-3xl shadow-2xl overflow-hidden border border-gray-100">
                    <div class="p-8 lg:p-12">
                        <div class="text-center mb-12">
                            <h2 class="text-3xl lg:text-4xl font-black text-gray-900 mb-4">
                                <span class="bg-gradient-to-r from-green-500 to-blue-500 bg-clip-text text-transparent">
                                    {{ __('Related Products') }}
                                </span>
                            </h2>
                            <div class="w-24 h-1 bg-gradient-to-r from-green-500 to-blue-500 mx-auto rounded-full"></div>
                        </div>

                        <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-8">
                            @foreach($relatedProducts as $relatedProduct)
                                <div class="group bg-white rounded-2xl shadow-lg overflow-hidden hover:shadow-2xl transform hover:-translate-y-2 transition-all duration-500 border border-gray-100">
                                    <div class="relative overflow-hidden">
                                        @if($relatedProduct->image)
                                            <img src="{{ Storage::url($relatedProduct->image) }}" alt="{{ $relatedProduct->name }}" class="w-full h-56 object-cover group-hover:scale-110 transition-transform duration-500">
                                        @else
                                            <div class="w-full h-56 bg-gradient-to-br from-gray-100 to-gray-200 flex items-center justify-center">
                                                <svg class="w-12 h-12 text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"/>
                                                </svg>
                                            </div>
                                        @endif
                                    </div>
                                    <div class="p-6">
                                        <h3 class="text-xl font-bold text-gray-900 mb-2 group-hover:text-blue-600 transition-colors duration-300">
                                            {{ $relatedProduct->name }}
                                        </h3>
                                        <p class="text-gray-600 text-sm mb-4 leading-relaxed">
                                            {{ Str::limit($relatedProduct->description, 80) }}
                                        </p>
                                        <div class="flex justify-between items-center mb-4">
                                            <span class="text-2xl font-black text-gray-900">
                                                ${{ number_format($relatedProduct->price, 2) }}
                                            </span>
                                        </div>
                                        <a href="{{ route('products.show', $relatedProduct) }}" class="block w-full text-center bg-gradient-to-r from-green-500 to-blue-500 hover:from-green-600 hover:to-blue-600 text-white font-semibold py-3 px-6 rounded-xl transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl">
                                            {{ __('View Details') }}
                                        </a>
                                    </div>
                                </div>
                            @endforeach
                        </div>
                    </div>
                </div>
            @endif
        </div>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const basePrice = {{ $product->hasDiscount() ? $product->discounted_price : $product->price }};
        let additionalPrice = 0;
        let selectedOptions = {};

        // Function to update total price
        function updateTotalPrice() {
            const totalPrice = basePrice + additionalPrice;
            const priceElements = document.querySelectorAll('.price-display');
            const additionalCostsDiv = document.getElementById('additional-costs');
            const additionalCostsText = document.getElementById('additional-costs-text');

            priceElements.forEach(element => {
                element.textContent = '$' + totalPrice.toFixed(2);
            });

            // Show/hide additional costs
            if (additionalPrice > 0) {
                additionalCostsDiv.classList.remove('hidden');
                additionalCostsText.textContent = `+ $${additionalPrice.toFixed(2)} additional costs`;
            } else {
                additionalCostsDiv.classList.add('hidden');
            }

            // Update the hidden input for form submission
            const totalPriceInput = document.getElementById('total_price');
            if (totalPriceInput) {
                totalPriceInput.value = totalPrice.toFixed(2);
            }

            // Update dynamic selections
            const dynamicSelectionsInput = document.getElementById('dynamic_selections');
            if (dynamicSelectionsInput) {
                dynamicSelectionsInput.value = JSON.stringify(selectedOptions);
            }
        }

        // Handle select fields
        document.querySelectorAll('.dynamic-field-select').forEach(select => {
            select.addEventListener('change', function() {
                const selectedOption = this.options[this.selectedIndex];
                const price = parseFloat(selectedOption.dataset.price) || 0;
                const fieldIndex = this.dataset.fieldIndex;

                // Remove previous price for this field
                const previousPrice = parseFloat(this.dataset.previousPrice) || 0;
                additionalPrice -= previousPrice;

                // Add new price
                additionalPrice += price;
                this.dataset.previousPrice = price;

                // Update selections
                if (selectedOption.value) {
                    selectedOptions[fieldIndex] = {
                        type: 'select',
                        value: selectedOption.value,
                        price: price
                    };
                } else {
                    delete selectedOptions[fieldIndex];
                }

                updateTotalPrice();
            });
        });

        // Handle checkbox fields
        document.querySelectorAll('.dynamic-field-checkbox').forEach(checkbox => {
            checkbox.addEventListener('change', function() {
                const price = parseFloat(this.dataset.price) || 0;
                const fieldName = this.name;
                const fieldIndex = fieldName.match(/dynamic_field_(\d+)/)[1];

                if (this.checked) {
                    additionalPrice += price;

                    // Initialize array if not exists
                    if (!selectedOptions[fieldIndex]) {
                        selectedOptions[fieldIndex] = {
                            type: 'checkbox',
                            values: [],
                            price: 0
                        };
                    }

                    selectedOptions[fieldIndex].values.push(this.value);
                    selectedOptions[fieldIndex].price += price;
                } else {
                    additionalPrice -= price;

                    if (selectedOptions[fieldIndex]) {
                        const valueIndex = selectedOptions[fieldIndex].values.indexOf(this.value);
                        if (valueIndex > -1) {
                            selectedOptions[fieldIndex].values.splice(valueIndex, 1);
                            selectedOptions[fieldIndex].price -= price;
                        }

                        if (selectedOptions[fieldIndex].values.length === 0) {
                            delete selectedOptions[fieldIndex];
                        }
                    }
                }

                updateTotalPrice();
            });
        });

        // Handle radio fields
        document.querySelectorAll('.dynamic-field-radio').forEach(radio => {
            radio.addEventListener('change', function() {
                const price = parseFloat(this.dataset.price) || 0;
                const fieldName = this.name;
                const fieldIndex = fieldName.match(/dynamic_field_(\d+)/)[1];

                // Remove previous price for this field group
                const previousChecked = document.querySelector(`input[name="${fieldName}"]:checked`);
                if (previousChecked && previousChecked !== this) {
                    const previousPrice = parseFloat(previousChecked.dataset.price) || 0;
                    additionalPrice -= previousPrice;
                }

                // Add new price
                additionalPrice += price;

                // Update selections
                selectedOptions[fieldIndex] = {
                    type: 'radio',
                    value: this.value,
                    price: price
                };

                updateTotalPrice();
            });
        });

        // Initialize price display
        updateTotalPrice();
    });
    </script>
</x-app-layout>

@push('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        const thumbnails = document.querySelectorAll('.thumbnail-image');
        const mainImage = document.getElementById('main-product-image');

        thumbnails.forEach(thumbnail => {
            thumbnail.addEventListener('click', function() {
                const imageUrl = this.getAttribute('data-image');
                if (mainImage && imageUrl) {
                    mainImage.src = imageUrl;
                }
            });
        });
    });
</script>
@endpush
